<?php
namespace betawall\helpers;

use betawall\api\data\MalwareGetUpdateRequestData;
use betawall\api\data\MalwareSendLogsData;
use betawall\api\data\MalwareSendReportData;
use betawall\api\data\MalwareSendRequestData;
use betawall\api\requests\MalwareGetUpdateRequest;
use betawall\api\requests\MalwareSendRequest;
use betawall\db\models\MalwareModel;
use betawall\db\models\SettingsModel;

class TasksHelper
{

    const BW_SCHEDULE_MALWARE_SEND = 'bw_schedule_malware_send';
    const BW_SCHEDULE_MALWARE_GET_UPDATE = 'bw_schedule_malware_get_update';

    const INTERVAL_DAILY = 'daily';
    const INTERVAL_HOURLY = 'hourly';
    const INTERVAL_TWICE_DAILY = 'twicedaily';
    const INTERVAL_WEEKLY = 'weekly';

    public function malware_send()
    {
        $malwareModel = new MalwareModel();
        $malwareStat = $malwareModel->get_all_with_ip_check_in_blacklist(MalwareModel::SEND_PORTION_LIMIT);

        if (!empty($malwareStat)) {
            $statIds = [];
            $groupedStat = [];

            foreach ($malwareStat as $item) {
                $statIds[] = $item->id;

                if (isset($groupedStat[$item->ip])) {
                    switch ($item->request_type) {
                        case MalwareModel::TYPE_NOT_FOUND:
                            $groupedStat[$item->ip]['type_1']++;
                        break;
                        case MalwareModel::TYPE_LOGIN:
                            $groupedStat[$item->ip]['type_2']++;
                        break;
                        case MalwareModel::TYPE_SQL_INJECTION:
                            $groupedStat[$item->ip]['type_3']++;
                        break;
                        case MalwareModel::TYPE_CODE_EXECUTION:
                            $groupedStat[$item->ip]['type_4']++;
                        break;
                        case MalwareModel::TYPE_XSS:
                            $groupedStat[$item->ip]['type_5']++;
                        break;
                        case MalwareModel::TYPE_RFI:
                            $groupedStat[$item->ip]['type_6']++;
                        break;
                        case MalwareModel::TYPE_SUSPICIOUS_FILE:
                            $groupedStat[$item->ip]['type_7']++;
                        break;
                    }

                    $groupedStat[$item->ip]['logs'][] = [
                        'request_type' => $item->request_type,
                        'user_agent' => $item->user_agent,
                        'info' => $item->logs,
                        'incoming_at' => $item->created_at
                    ];
                } else {
                    $groupedStat[$item->ip] = [
                        'type_1' => ($item->request_type == MalwareModel::TYPE_NOT_FOUND) ? 1 : 0,
                        'type_2' => ($item->request_type == MalwareModel::TYPE_LOGIN) ? 1 : 0,
                        'type_3' => ($item->request_type == MalwareModel::TYPE_SQL_INJECTION) ? 1 : 0,
                        'type_4' => ($item->request_type == MalwareModel::TYPE_CODE_EXECUTION) ? 1 : 0,
                        'type_5' => ($item->request_type == MalwareModel::TYPE_XSS) ? 1 : 0,
                        'type_6' => ($item->request_type == MalwareModel::TYPE_RFI) ? 1 : 0,
                        'type_7' => ($item->request_type == MalwareModel::TYPE_SUSPICIOUS_FILE) ? 1 : 0,
                        'logs' => [
                            [
                                'request_type' => $item->request_type,
                                'user_agent' => $item->user_agent,
                                'info' => $item->logs,
                                'incoming_at' => $item->created_at,
                            ]
                        ]
                    ];
                }
            }

            if (!empty($groupedStat)) {
                $reports = [];

                foreach ($groupedStat as $key => $value) {
                    $logs = [];

                    foreach ($value['logs'] as $log) {
                        $logs[] = new MalwareSendLogsData(
                            $log['request_type'],
                            $log['user_agent'],
                            $log['info'],
                            $log['incoming_at']
                        );
                    }

                    $reports[] = new MalwareSendReportData(
                        $key,
                        $value['type_1'],
                        $value['type_2'],
                        $value['type_3'],
                        $value['type_4'],
                        $value['type_5'],
                        $value['type_6'],
                        $value['type_7'],
                        $logs
                    );
                }

                if (!empty($reports)) {
                    $settingsModel = new SettingsModel();
                    $siteUrl = MainHelper::getSiteUrl();
                    $apiKey = $settingsModel->get_by_name($settingsModel->secret_key);

                    $requestData = new MalwareSendRequestData($siteUrl, $reports);
                    $request = new MalwareSendRequest($requestData, $apiKey->value_text ?? '');
                    $response = $request->sendRequest();

                    if ($response->success && $malwareModel->update_is_sent_status($statIds)) {
                        $malwareModel->delete_by_conditions([
                            $malwareModel->malware_structure->is_sent => MalwareModel::STATUS_SENT
                        ], 1000);
                    }

                    return $response;
                }
            }
        }

        return false;
    }

    public function malware_get_update()
    {
        $settingsModel = new SettingsModel();
        $siteUrl = MainHelper::getSiteUrl();
        $apiKey = $settingsModel->get_by_name($settingsModel->secret_key);

        $requestData = new MalwareGetUpdateRequestData($siteUrl);
        $request = new MalwareGetUpdateRequest($requestData, $apiKey->value_text ?? '');
        $response = $request->sendRequest();

        $binaryHelper = BinaryHelper::getInstance();

        if ($response->success) {
            $permissionsAccess = $binaryHelper->checkPermissionsAccess();

            if ($response->db_link && $request->downloadDb($response->db_link)) {
                $wp_filesystem = FileSystemHelper::get_filesystem();
                $archivePath = BinaryHelper::BINARY_DB_DIRECTORY . '/' . BinaryHelper::BINARY_DB_UPDATE_ARCHIVE;

                if ($wp_filesystem->exists($archivePath)) {
                    $binaryHelper->extractTarGz($archivePath);
                }
            }

            if ($response->deactivated) {
                $settingsModel->update_by_name($settingsModel->error_states, [
                    $settingsModel->settings_structure->value_number => SettingsModel::ERROR_STATE_DEACTIVATED
                ]);
            } elseif ($response->unverified) {
                $settingsModel->update_by_name($settingsModel->error_states, [
                    $settingsModel->settings_structure->value_number => SettingsModel::ERROR_STATE_NOT_VERIFIED
                ]);
            } elseif ($response->request_problems) {
                $settingsModel->update_by_name($settingsModel->error_states, [
                    $settingsModel->settings_structure->value_number => SettingsModel::ERROR_STATE_WRONG_DOMAIN
                ]);
            } else {
                $settingsModel->update_by_name($settingsModel->error_states, [
                    $settingsModel->settings_structure->value_number => SettingsModel::ERROR_STATE_NO_ERRORS
                ]);
            }

            if (!$permissionsAccess) {
                $settingsModel->update_by_name($settingsModel->access_state, [
                    $settingsModel->settings_structure->value_number => SettingsModel::ACCESS_STATE_DENIED
                ]);
            } else {
                $settingsModel->update_by_name($settingsModel->access_state, [
                    $settingsModel->settings_structure->value_number => SettingsModel::ACCESS_STATE_ALLOWED
                ]);
            }

            $binaryHelper->setS($response->s);
        } else if (isset($response->data['code']) && $response->data['code'] === 401) {
            $settingsModel->update_by_name($settingsModel->error_states, [
                $settingsModel->settings_structure->value_number => SettingsModel::ERROR_STATE_NOT_VERIFIED
            ]);
            $binaryHelper->setS();
        }

        return $response;
    }

}