<?php
namespace betawall\helpers;
use RecursiveIteratorIterator;
use RecursiveDirectoryIterator;

class SecurityHelper
{
    private static function folder_to_file_type(): array
    {
        $parts = explode('/', WP_PLUGIN_DIR);
        $pluginsDirName = end($parts);

        return [
            'themes'        => 'Theme File',
            $pluginsDirName => 'Plugin File',
            'mu-plugins'    => 'MU Plugin File',
        ];
    }

    private static function get_plugin_path_to_exclude(): string
    {
        return realpath(WP_PLUGIN_DIR . '/' . BW_PLUGIN_SLUG);
    }

    public static function detect_hidden_users()
    {
        global $wpdb;

        $wp_users = get_users(['fields' => ['ID']]);
        $wp_user_ids = wp_list_pluck($wp_users, 'ID');

        $raw_users = $wpdb->get_results(
            "SELECT ID, user_login, user_email, display_name, user_registered
            FROM {$wpdb->users}
            WHERE user_status = 0",
            ARRAY_A
        );

        $hidden_users = [];
        foreach ($raw_users as $user) {
            $id = $user['ID'];
            if (!in_array($id, $wp_user_ids)) {
                $meta_keys = $wpdb->get_results(
                    $wpdb->prepare(
                        "SELECT meta_key, meta_value FROM {$wpdb->usermeta} WHERE user_id = %d",
                        $id
                    )
                );

                $lastLogin = 'N/A (possible programmatic access)';
                $meta_keys_raw = [];
                foreach ($meta_keys as $meta_key) {
                    if ($meta_key->meta_key == 'wfls-last-login') {
                        $lastLogin = $meta_key->meta_value ? date('Y-m-d H:i:s', $meta_key->meta_value) : $lastLogin;
                    }
                    $meta_keys_raw[] = $meta_key->meta_key;
                }

                $hidden_users[] = [
                    'ID'              => $id,
                    'user_login'      => $user['user_login'],
                    'user_email'      => $user['user_email'],
                    'display_name'    => $user['display_name'],
                    'user_registered' => $user['user_registered'],
                    'last_login'      => $lastLogin,
                    'meta_keys'       => $meta_keys_raw,
                ];
            }
        }

        return $hidden_users;
    }


    public static function find_user_creators(array $hiddenUsers = []): array
    {
        global $wpdb;
        $raw_prefix = $wpdb->prefix;
        $prefix = preg_quote( $raw_prefix, '/' );

        $helpFileInfoArray = self::security_help_file_info();

        $base_dirs = [
            WP_CONTENT_DIR . '/themes',
            WP_PLUGIN_DIR,
            WP_CONTENT_DIR . '/mu-plugins',
        ];

        $results = [];
        $exclude = self::get_plugin_path_to_exclude();

        $quickPatterns = [
            'wp_create_user(',
            'wp_insert_user(',
            'INSERT',
            '$wpdb',
        ];

        foreach ($base_dirs as $dir) {
            if (!is_dir($dir)) {
                continue;
            }

            $iterator = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS)
            );

            foreach ($iterator as $file) {
                $path = $file->getPathname();
                if (
                    !isset($helpFileInfoArray[$path]) ||
                    strpos($path, $exclude) === 0 ||
                    $file->getExtension() !== 'php'
                ) {
                    continue;
                }

                $hasPattern = false;
                foreach ($quickPatterns as $entry) {
                    if (isset($helpFileInfoArray[$path][$entry])) {
                        $hasPattern = true;
                        break;
                    }
                }

                if (!$hasPattern) {
                    continue;
                }

                $code = file_get_contents($path);
                if (!$code) {
                    continue;
                }

                $matchedUser = null;
                foreach ($hiddenUsers as $u) {
                    if (
                        stripos($code, $u['user_login']) !== false &&
                        stripos($code, $u['user_email']) !== false
                    ) {
                        $matchedUser = $u;
                        break;
                    }
                }

                if (!$matchedUser) {
                    continue;
                }
                unset($matchedUser['meta_keys']);

                $collect1 = function ($method, $snippet) use (&$results, $file, $path, $matchedUser) {
                    preg_match('~/' . preg_quote($_SERVER['SERVER_NAME'], '/') . '/wp-content/(.*)/~U', $path, $q);
                    $folder = $q[1] ?? '';

                    $helpArr = explode($_SERVER['SERVER_NAME'], $path);
                    $file = end($helpArr);
                    $results[] = [
                        'file'        => $file,
                        'file_orig'   => $path,
                        'folder'      => $folder,
                        'file_type'   => self::folder_to_file_type()[$folder] ?? 'unknown File',
                        'method'      => $method,
                        'call'        => trim($snippet),
                        'source_user' => $matchedUser,
                    ];
                };

                $patterns = [
                    'wp_create_user()'          => '/\bwp_create_user\s*\(/is',
                    'wp_insert_user()'          => '/\bwp_insert_user\s*\(/is',
                    '$wpdb->insert()'           => '/\$wpdb\s*->\s*insert\s*\(/is',
                    '$wpdb->query()->prepare()' => '/\$wpdb\s*->\s*query\s*\(\s*\$wpdb\s*->\s*prepare\s*\(\s*INSERT\s+INTO\s+[`\'"]?'. $prefix .'users[`\'"]?/is',
                    'sql_("INSERT '. $prefix .'users")' => '/INSERT\s+INTO\s+[`\'"]?'. $prefix .'users[`\'"]?/is',
                ];

                foreach ($patterns as $name => $regex) {
                    if (preg_match($regex, $code, $m)) {
                        $collect1($name, $m[0]);
                        break;
                    }
                }
            }
        }

        return $results;
    }

    public static function scan_hooks_for_hidden_users(array $hiddenUsers = []): array
    {
        global $wpdb;
        $raw_prefix = $wpdb->prefix;
        $prefix = preg_quote( $raw_prefix, '/' );

        $helpFileInfoArray = self::security_help_file_info();

        $base_dirs = [
            WP_CONTENT_DIR . '/themes',
            WP_PLUGIN_DIR,
            WP_CONTENT_DIR . '/mu-plugins',
        ];

        $exclude = self::get_plugin_path_to_exclude();
        $results = [];

        $quickPatterns = [
            'add_action(',
            'add_filter(',
            'pre_user_query',
            'get_users',
            'user_login',
            'user_email',
            '$wpdb',
            'query(',
            'prepare(',
            'user_pass',
        ];

        foreach ($base_dirs as $dir) {
            if (!is_dir($dir)) continue;

            $iterator = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS)
            );

            foreach ($iterator as $file) {
                $path = realpath($file->getPathname());

                if (
                    !isset($helpFileInfoArray[$path]) ||
                    strpos($path, $exclude) === 0 ||
                    $file->getExtension() !== 'php'
                ) {
                    continue;
                }

                $hasPattern = false;
                foreach ($quickPatterns as $entry) {
                    if (isset($helpFileInfoArray[$path][$entry])) {
                        $hasPattern = true;
                        break;
                    }
                }

                if (!$hasPattern) {
                    continue;
                }

                $code = file_get_contents($path);
                if (!$code) continue;

                $code = preg_replace('/\/\*.*?\*\/|\/\/.*?$|#.*?$/ms', '', $code);

                $matchedUser = null;
                foreach ($hiddenUsers as $u) {
                    if (
                        stripos($code, $u['user_login']) !== false ||
                        stripos($code, $u['user_email']) !== false ||
                        stripos($code, (string)$u['ID']) !== false
                    ) {
                        $matchedUser = $u;
                        break;
                    }
                }
                if (!$matchedUser) {
                    continue;
                }
                unset($matchedUser['meta_keys']);

                $collect = function ($method, $snippet) use (&$results, $file, $path, $matchedUser) {
                    preg_match('~/' . preg_quote($_SERVER['SERVER_NAME'], '/') . '/wp-content/(.*)/~U', $path, $q);
                    $folder = $q[1] ?? '';

                    $helpArr = explode($_SERVER['SERVER_NAME'], $path);
                    $file = end($helpArr);

                    $results[] = [
                        'file'        => $file,
                        'file_orig'   => $path,
                        'folder'      => $folder,
                        'file_type'   => self::folder_to_file_type()[$folder] ?? 'unknown File',
                        'method'      => $method,
                        'call'        => trim($snippet),
                        'source_user' => $matchedUser,
                    ];
                };

                $patterns = [
                    'pre_user_query hook injection' =>
                        '/add_(?:action|filter)\s*\(\s*[\'"]pre_user_query[\'"]\s*,\s*function\s*\((.*?)\)\s*\{([\s\S]*?)\}/is',

                    'direct $query->query_where injection' =>
                        '/->\s*query_where\s*\.\=\s*[\'"][^\'"]+[\'"]/is',

                    '$wpdb->get_results() + user_login/email filter' =>
                        '/\$wpdb\s*->\s*get_results\s*\(\s*\$wpdb\s*->\s*prepare\s*\([\s\S]*?(user_login|user_email)[\s\S]*?\)/is',

                    'SELECT FROM users WHERE user_login !=' =>
                        '/SELECT\s+[\s\S]*?FROM\s+[\'"]?('. $prefix .'users|\$[A-Za-z_][A-Za-z0-9_]*)[\'"]?\s*WHERE\s+[\s\S]*?(user_login|user_email)\s*(!=|NOT\s*IN)/is',
                ];

                foreach ($patterns as $label => $regex) {
                    if (preg_match($regex, $code, $m)) {
                        $collect($label, $m[0]);
                        break;
                    }
                }
            }
        }

        return $results;
    }

    /**
     * Сканирует плагины на предмет сокрытия.
     *
     * @return array
     */
    public static function scan_plugin_hiding(): array
    {
        global $wpdb;

        $helpFileInfoArray = self::security_help_file_info();
        $exclude = self::get_plugin_path_to_exclude();

        $all_plugins = get_plugins();
        $mu_plugins = get_mu_plugins();
        $found = [];

        $dirs = [
            WP_CONTENT_DIR . '/mu-plugins',
            WP_PLUGIN_DIR,
        ];

        $allPluginsHelp = [];
        foreach ($all_plugins as $pluginPath => $plugin) {
            $allPluginsHelp[$plugin['Name']] = $plugin;
            $allPluginsHelp[$plugin['Name']]['pluginPath'] = $pluginPath;
        }

        $quickPatterns = [
            'add_filter(',
            'all_plugins',
            'get_plugins(',
            'option_active_plugins',
            'basename(',
            'data-plugin=',
        ];

        foreach ($dirs as $dir) {
            if (!is_dir($dir)) {
                continue;
            }

            $it = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS)
            );

            foreach ($it as $file) {
                $path = $file->getPathname();

                if (!isset($helpFileInfoArray[$path])) {
                    continue;
                }

                if (strpos($path, $exclude) === 0) {
                    continue;
                }

                $foundEntries = false;
                foreach ($quickPatterns as $quickPattern) {
                    if (isset($helpFileInfoArray[$path][$quickPattern])) {
                        $foundEntries = true;
                        break;
                    }
                }

                if (!$foundEntries) {
                    continue;
                }

                $code = file_get_contents($path);
                if (!$code) {
                    continue;
                }

                $modifyDate = date('Y-m-d', $file->getMTime());

                $plugin_folder = 'unknown';
                if (strpos($path, WP_PLUGIN_DIR . '/') === 0) {
                    $rel           = substr($path, strlen(WP_PLUGIN_DIR) + 1);
                    $plugin_folder = strtok($rel, '/');
                } elseif (strpos($path, WP_CONTENT_DIR . '/mu-plugins/') === 0) {
                    $rel           = substr($path, strlen(WP_CONTENT_DIR . '/mu-plugins/') + 1);
                    $parts         = explode('/', $rel);
                    $plugin_folder = $parts[0];
                }

                if (preg_match('/add_filter\s*\(\s*[\'"]all_plugins[\'"]\s*,\s*(?:[\'"]([a-zA-Z0-9_]+)[\'"]|array\s*\(.*?,?\s*[\'"]([a-zA-Z0-9_]+)[\'"]\\s*\)|(function\s*\((\s*\$[a-zA-Z0-9_]+(?:\s*,\s*\$[a-zA-Z0-9_]+)*\s*)?\)\s*\{.*?\}))\s*\);?/s', $code, $m)) {
                    $callback_code = '';
                    $callback_param_name = '';

                    if (!empty($m[3])) {
                        $callback_code = $m[3];
                        if (preg_match('/function\s*\(\s*(\$[a-zA-Z0-9_]+)/', $m[4], $param_match)) {
                            $callback_param_name = $param_match[1];
                        }
                    } elseif (!empty($m[1]) || !empty($m[2])) {
                        $func_name = !empty($m[1]) ? $m[1] : $m[2];

                        $func_pattern = '/function\s+' . preg_quote($func_name, '/') . '\s*\(\s*(\$[a-zA-Z0-9_]+(?:\s*,\s*\$[a-zA-Z0-9_]+)*\s*)?\)\s*\{(.*)\}/s';

                        if (preg_match($func_pattern, $code, $func_body_current_file)) {
                            $callback_code = $func_body_current_file[2];
                            if (!empty($func_body_current_file[1])) {
                                if (preg_match('/\s*(\$[a-zA-Z0-9_]+)/', $func_body_current_file[1], $param_match)) {
                                    $callback_param_name = $param_match[1];
                                }
                            }
                        } else {
                            if (!empty($plugin_root_path) && is_dir($plugin_root_path)) {
                                $plugin_file_it = new RecursiveIteratorIterator(
                                    new RecursiveDirectoryIterator($plugin_root_path, RecursiveDirectoryIterator::SKIP_DOTS)
                                );
                                foreach ($plugin_file_it as $plugin_file_entry) {
                                    if ($plugin_file_entry->getExtension() !== 'php' || $plugin_file_entry->getPathname() === $path) {
                                        continue;
                                    }
                                    $plugin_file_code = file_get_contents($plugin_file_entry->getPathname());
                                    if ($plugin_file_code && preg_match($func_pattern, $plugin_file_code, $func_body_other_file)) {
                                        $callback_code = $func_body_other_file[2];
                                        if (!empty($func_body_other_file[1])) {
                                            if (preg_match('/\s*(\$[a-zA-Z0-9_]+)/', $func_body_other_file[1], $param_match)) {
                                                $callback_param_name = $param_match[1];
                                            }
                                        }
                                        break;
                                    }
                                }
                            }
                        }
                    }

                    $is_suspicious = false;
                    if (!empty($callback_code) && !empty($callback_param_name)) {
                        $param_escaped = preg_quote($callback_param_name, '/');

                        if (
                            preg_match('/' . $param_escaped . '\s*=\s*\[\s*\]\s*;/', $callback_code) ||
                            preg_match('/return\s*\[\s*\]\s*;/', $callback_code) ||
                            preg_match('/unset\s*\(\s*' . $param_escaped . '\s*\[/', $callback_code) ||
                            preg_match('/array_filter\s*\(\s*' . $param_escaped . '\s*,/', $callback_code) ||
                            preg_match('/array_diff_key\s*\(\s*' . $param_escaped . '\s*,/', $callback_code) ||
                            preg_match('/foreach\s*\(\s*' . $param_escaped . '\s*as\s*(?:\$[a-zA-Z0-9_]+\s*=>\s*)?(\$[a-zA-Z0-9_]+)\s*\)\s*\{.*?unset\s*\(\s*' . $param_escaped . '\s*\[(?:\$[a-zA-Z0-9_]+)?\s*\]\s*\)/is', $callback_code)
                        ) {
                            $is_suspicious = true;
                        }
                    } else {
                        $is_suspicious = true;
                    }

                    if ($is_suspicious) {
                        $found[] = [
                            'file' => $path,
                            'method' => 'filter: all_plugins',
                            'modify' => $modifyDate,
                            'plugin_folder' => $plugin_folder
                        ];
                    }
                }


                if (preg_match('/add_filter\s*\(\s*[\'"]option_active_plugins[\'"]/', $code)) {
                    $found[] = ['file' => $path, 'method' => 'filter: option_active_plugins', 'modify' => $modifyDate, 'plugin_folder' => $plugin_folder];
                }

                $opts_table = preg_quote($wpdb->options, '/');

                if (preg_match("/UPDATE\s+`?{$opts_table}`?\s+SET\s+`?option_value`?\s*=\s*REPLACE\s*\(/i", $code) && preg_match("/['\"]active_plugins['\"]/", $code)) {
                    $found[] = ['file' => $path, 'method' => 'SQL: update active_plugins', 'modify' => $modifyDate, 'plugin_folder' => $plugin_folder];
                }
                if (preg_match("/DELETE\s+FROM\s+`?{$opts_table}`?\s+WHERE\s+`?option_name`?\s*=\s*['\"]active_plugins['\"]/i", $code)) {
                    $found[] = ['file' => $path, 'method' => 'SQL: delete active_plugins', 'modify' => $modifyDate, 'plugin_folder' => $plugin_folder];
                }

                if (preg_match('/\$GLOBALS\s*\[\s*[\'"]wp_filter[\'"]\\s*\]\s*\[\s*[\'"]all_plugins[\'"]\s*\]\s*=\s*\[\s*\]\s*;/i', $code)) {
                    $found[] = ['file' => $path, 'method' => 'direct hook manipulation', 'modify' => $modifyDate, 'plugin_folder' => $plugin_folder];
                }

                $pattern = '/add_action\s*\(\s*[\'"](admin_footer|admin_head)[\'"]\s*,\s*function\s*\([^)]*\)\s*\{([\s\S]*?)\}\s*\)/xi';

                if (preg_match($pattern, $code, $match)) {
                    $js_code = $match[2];

                    $js_pattern = '/tr\s*\[\s*[\'"`]?data-plugin[\'"`]?\s*=[\s\S]*?\.(?:hide|remove)\s*\(\)|tr\s*\[\s*[\'"`]?data-plugin[\'"`]?\s*=[\s\S]*?\.style\.display\s*=\s*[\'"`]none[\'"`]/i';

                    if (preg_match($js_pattern, $js_code, $m_js)) {
                        $found[] = [
                            'file'          => $path,
                            'method'        => 'JS hide plugin row',
                            'modify'        => $modifyDate,
                            'plugin_folder' => $plugin_folder,
                        ];
                    }
                }
            }
        }

        $unique = [];
        foreach ($found as $row) {
            $key = ($row['file'] ?? '') . '||' . ($row['method'] ?? '');
            if (!isset($unique[$key])) {
                $unique[$key] = $row;
            }
        }
        $list = array_values($unique);

        $groups = [];
        foreach ($list as $row) {
            $folder = $row['plugin_folder'] ?? 'unknown';
            if (!isset($groups[$folder])) {
                $groups[$folder] = ['modify' => '1970-01-01', 'hits' => []];
            }
            if ($groups[$folder]['modify'] < $row['modify']) {
                $groups[$folder]['modify'] = $row['modify'];
            }
            $groups[$folder]['hits'][] = $row;
        }

        $final_groups = [];
        foreach ($groups as $folder => $data) {
            $current_plugin_name = $folder;
            $current_plugin_version = null;
            $current_plugin_author = null;
            $current_plugin_path = null;
            $current_full_fs_path = null;
            $current_textdomain = null;

            $found_meta = false;

            foreach ($all_plugins as $key => $meta) {

                if (strpos($key, $folder . '/') === 0) {
                    $found_meta = true;
                } elseif (strpos($key, 'mu-plugins/') === 0 && basename($key, '.php') === $folder) {
                    $found_meta = true;
                }
                if ($found_meta) {
                    $current_plugin_name    = $meta['Name'];
                    $current_plugin_version = $meta['Version'];
                    $current_plugin_author  = $meta['Author'];
                    $current_textdomain     = $meta['TextDomain'];
                    $current_plugin_path    = $key;
                    $current_full_fs_path   = WP_PLUGIN_DIR . '/' . $key;
                    if (strpos($key, 'mu-plugins/') === 0) {
                        $current_full_fs_path = WP_CONTENT_DIR . '/' . $key;
                    }
                    break;
                }
            }

            if (!$found_meta && $folder !== 'unknown' && !empty($data['hits'])) {
                $first_hit_file = $data['hits'][0]['file'];
                if(strpos($first_hit_file, WP_CONTENT_DIR . '/mu-plugins/') === 0) {
                    $current_plugin_name = basename($first_hit_file);
                    $current_plugin_path = 'mu-plugins/' . basename($first_hit_file);
                } else {
                    $current_plugin_name = $folder . ' (No header)';
                    $current_plugin_path = $folder;
                }
                $current_full_fs_path = $first_hit_file;
            }

            $final_groups[$folder] = array_merge($data, [
                'Name'         => $current_plugin_name,
                'Version'      => $current_plugin_version,
                'Author'       => $current_plugin_author,
                'TextDomain'   => $current_textdomain,
                'plugin_path'  => $current_plugin_path,
                'full_fs_path' => $current_full_fs_path,
                'mu_plugin_path' => '',
            ]);
        }

        foreach ($final_groups as $folder => $final_group) {
            if (strpos($final_group['plugin_path'], 'mu-plugins/') === 0) {
                foreach ($mu_plugins as $file => $meta) {
                    if (strtolower($file) === strtolower(str_replace('mu-plugins/', '', $final_group['plugin_path']))) {
                        $final_groups[$folder]['mu_plugin_path'] = $final_group['plugin_path'];
                        $existingPlugin = $allPluginsHelp[$meta['Name']] ?? [];
                        if ($existingPlugin) {
                            $final_groups[$folder]['plugin_path'] = $existingPlugin['pluginPath'];
                            $final_groups[$folder]['Name'] = $existingPlugin['Name'] ?: $final_groups[$folder]['Name'];
                        }

                        break;
                    }
                }
            }
        }

        return $final_groups;
    }

    public static function security_help_file_info()
    {
		$wp_filesystem = FileSystemHelper::get_filesystem();
		if ($wp_filesystem && !$wp_filesystem->is_dir(BW_BACKUPS_DIR)) {
			$wp_filesystem->mkdir(BW_BACKUPS_DIR, 0777);
		}
		
        $filePath = BW_BACKUPS_DIR . 'security_help_file_info.php';
        if ($wp_filesystem && $wp_filesystem->is_dir(BW_BACKUPS_DIR) && file_exists($filePath)) {
            $data = @json_decode(ltrim(file_get_contents($filePath), '<?php //'), true);
            if (time() - 300 < ($data['time'] ?? 0)) {
                return $data['files'];
            }
        }

        $muPluginsDir = WP_CONTENT_DIR . '/mu-plugins';
        $base_dirs = [
            WP_CONTENT_DIR . '/themes',
            WP_PLUGIN_DIR,
            $muPluginsDir,
        ];

        $exclude  = self::get_plugin_path_to_exclude();

        $filesWithEntries = [];

        $quickPatterns = [
            'wp_create_user(',
            'wp_insert_user(',
            'INSERT',
            'SELECT',
            '$wpdb',
            'query(',
            'prepare(',
            'user_login',
            'user_email',
            'user_pass',
            'user_nicename',
            'usermeta',
            'add_filter(',
            'pre_user_query',
            'pre_get_users',
            'all_plugins',
            'option_active_plugins',
            'get_plugins(',
            'data-plugin=',
            'basename(',
        ];

        $userCreationQuickPatterns = [
            'wp_create_user(' => 'wp_create_user(',
            'wp_insert_user(' => 'wp_insert_user(',
            'INSERT' => 'INSERT',
            '$wpdb' => '$wpdb',
        ];

        $essentialRequieredKeys = [
            'user_login',
            'wp_create_user(',
            'wp_insert_user(',
            'pre_user_query',
            'all_plugins',
            'data-plugin=',
            'get_plugins',
        ];

        $safeAlone = ['SELECT', 'query(', 'basename('];

        foreach ($base_dirs as $dir) {
            if (!is_dir($dir)) {
                continue;
            }
            $iterator = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS)
            );

            foreach ($iterator as $file) {
                $path = $file->getPathname();

                if (strpos($path, $exclude) === 0) {
                    continue;
                }

                if ($file->getExtension() !== 'php') {
                    continue;
                }

                $code = file_get_contents($path);

                $found = [];
                $userCreationPattern = false;
                foreach ($quickPatterns as $substr) {
                    if (stripos($code, $substr) !== false) {
                        $found[$substr] = true;
                    }
                    if (isset($userCreationQuickPatterns[$substr])) {
                        $userCreationPattern = true;
                    }
                }

                if (!$found) {
                    continue;
                }

                $foundKeys = array_keys($found);

                if (count($foundKeys) === 1 && in_array($foundKeys[0], $safeAlone, true)) {
                    continue;
                }

                if (empty(array_diff($foundKeys, $safeAlone))) {
                    continue;
                }

                if ($userCreationPattern && $dir !== $muPluginsDir) {
                    $hasUserData = false;
                    foreach ($essentialRequieredKeys as $key) {
                        if (stripos($code, $key) !== false) {
                            $hasUserData = true;
                            break;
                        }
                    }
                    if (!$hasUserData) {
                        continue;
                    }
                }

                $filesWithEntries['files'][$path] = $found;
            }
        }

        $filesWithEntries['time'] = time();

        @$wp_filesystem->put_contents($filePath, '<?php //' . json_encode($filesWithEntries));

        return $filesWithEntries['files'];
    }
}
