<?php
namespace betawall\helpers;

use betawall\components\NavManager;
use betawall\db\models\SettingsModel;

class MainHelper
{
    const TYPE_REST_API = 0;
    const TYPE_XML_RPC  = 1;
    const TYPE_RSS      = 2;

    const TYPE_TO_REGEXP = [
        self::TYPE_REST_API => "~(register_rest_route\(|add_action\(\s*['\"]\s*rest_api_init\s*['\"])~i",
        self::TYPE_XML_RPC  => "~(add_filter\(\s*['\"]xmlrpc_methods['\"]|do_action\(\s*['\"]xmlrpc_call['\"])~i",
        self::TYPE_RSS      => '~\b(get_feed\(|fetch_feed\(|wp_rss\(|do_feed\(|add_feed\(|the_permalink_rss\(|the_title_rss\(|the_content_feed\(|get_post_modified_time\(|get_lastpostmodified\(|wp_get_rss\(|do_feed_rss2|rss2_head|rss2_item)~i',
    ];

    const SUPPORT_LINK = 'https://account.betawall.pro/help';

    public $wp_filesystem;
    public $wpStoragePath;

    public function __construct()
    {
        $this->wp_filesystem = FileSystemHelper::get_filesystem();
    }

    public static function getSiteUrl(): string
    {
        return get_home_url();
    }

    public static function getSalt(): string
    {
        return wp_generate_password(10, true, true);
    }

    public static function getUnblockLink($unblockKey): string
    {
        return FirewallHelper::UNBLOCK_URI . $unblockKey;
    }

    public static function getSupportLink(): string
    {
        return self::SUPPORT_LINK;
    }

    public static function generateSecretKey(): string
    {
        return md5(self::getSiteUrl() . self::getSalt() . time());
    }

    public static function checkValidSecretKey($secretKey): int
    {
        return preg_match('/^[a-f0-9]{32}$/i', $secretKey);
    }

    public static function generateNonce(): string
    {
        return substr(wp_hash(wp_nonce_tick() . 'bw_nonce', 'nonce'), -12, 10);
    }

    public static function verifyNonce($nonce): bool
    {
        return $nonce === self::generateNonce();
    }

    public static function getClientIp(): string
    {
        if (!empty($_SERVER['HTTP_CF_CONNECTING_IP'])) {
            return $_SERVER['HTTP_CF_CONNECTING_IP'];
        }

        if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
            return trim($ips[0]);
        }

        return $_SERVER['REMOTE_ADDR'];
    }

    public static function isBase64Encoded(string $data): bool
    {
        return preg_match('%^[a-zA-Z0-9/+]*={0,2}$%', $data) && base64_decode($data, true) !== false;
    }

    public static function checkForAjaxRequestAccess()
    {
        $isAjaxRequest = self::checkIsAjaxRequest();
        $isAdminsRole = self::user_can_affect_site();

        if ($isAjaxRequest && $isAdminsRole) {
            return true;
        }

        if ($isAjaxRequest && $unblockKey = sanitize_text_field(trim($_REQUEST['unblock_key'] ?? ''))) {
            if (!self::checkValidSecretKey($unblockKey)) {
                return false;
            }
            $settingsModel = new SettingsModel();
            $secretKeyCurrent = $settingsModel->get_by_name($settingsModel->unblock_key);
            if ($secretKeyCurrent && $secretKeyCurrentValue = $secretKeyCurrent->value_text) {
                return $secretKeyCurrentValue === $unblockKey;
            }
        }

        return false;
    }

    public static function checkIsAjaxRequest(): bool
    {
        return !(!isset($_SERVER['HTTP_X_REQUESTED_WITH']) ||
            $_SERVER['HTTP_X_REQUESTED_WITH'] !== 'XMLHttpRequest');
    }

    public function checkSiteSSL(string $siteUrl): bool
    {
        if (strpos($siteUrl, 'https://') !== 0) {
            return false;
        }

        $response = wp_remote_get($siteUrl);
        if (is_wp_error($response)) {
            return false;
        }

        $response_code = wp_remote_retrieve_response_code($response);

        return $response_code === 200;
    }

    public function detectReadmeAndLicense(): array
    {
        $readmeFile = ABSPATH . 'readme.html';
        $licenseFile = ABSPATH . 'license.txt';

        $result = [
            'readme' => false,
            'license' => false
        ];

        if ($this->wp_filesystem->exists($readmeFile)) {
            $result['readme'] = true;
        }

        if ($this->wp_filesystem->exists($licenseFile)) {
            $result['license'] = true;
        }

        return $result;
    }

    public function detectDebugLog(): array
    {
        $configFile = ABSPATH . 'wp-config.php';

        $isEnabled = false;
        if ($this->wp_filesystem->exists($configFile) && $content = $this->wp_filesystem->get_contents($configFile)) {
            $commentedRegex = "/(?:\/\/|\/\*+|\*+)\s*define\s*\(\s*[\"']WP_DEBUG_LOG[\"']\s*,\s*(true|false)\s*\s*\)\s*;.*?(?:\*\/)?/s";
            $activeRegex = "/define\s*\(\s*[\"']WP_DEBUG_LOG[\"']\s*,\s*(true|false)\s*\s*\)\s*;/";

            if (preg_match($commentedRegex, $content, $matches)) {
                $isEnabled = false;
            } else if (preg_match($activeRegex, $content, $matches)) {
                $isEnabled = $matches[1] === 'true';
            }
        }

        $debugFile = ABSPATH . 'wp-content/debug.log';
        $existsDebugFile = $this->wp_filesystem->exists($debugFile);

        return [
            'is_enabled' => $isEnabled,
            'exists_debug_file' => $existsDebugFile
        ];
    }

    public function checkIsEnabledRestApi(): bool
    {
        $response = wp_remote_get(rest_url());

        return !is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200;
    }

    public function checkForExternalUsage(int $type): array
    {
        $allFiles = [];

        $pluginsDirPath = WP_PLUGIN_DIR;
        $activePlugins = $this->getActivePlugins();

        foreach ($activePlugins as $plugin) {
            $allFiles[$plugin] = $this->getAllPhpFilesInDirectory($pluginsDirPath . '/' . $plugin);
        }

        $externalUsage = [];
        $regexp = self::TYPE_TO_REGEXP[$type];
        foreach ($allFiles as $pluginName => $files) {
            $pluginName = ucwords(str_replace(['-', '_'], ' ', $pluginName));
            foreach ($files as $filePath) {
                $body = $this->wp_filesystem->get_contents($filePath);
                $match = [];
                preg_match_all($regexp, $body, $match);
                if (count($match[0] ?? [])) {
                    $externalUsage[] = $pluginName;
                    break;
                }
            }
        }

        return $externalUsage;
    }

    public function getActivePlugins(): array
    {
        $plugins = [];
        $pluginsDb = get_option('active_plugins');

        foreach($pluginsDb as $plugin) {
            $plugin = explode('/', $plugin)[0];

            if ($plugin !== 'betawall') {
                $plugins[] = explode('/', $plugin)[0];
            }
        }

        return $plugins;
    }

    public function getAllPhpFilesInDirectory(string $dir, bool $structureChecking = false): array
    {
        if (!$this->wpStoragePath) {
            $this->wpStoragePath = $dir . '/wp-content';
        }

        $files = [];
        $dirs = $this->wp_filesystem->dirlist($dir);

        $homePath = get_home_path();

        $coreAdminFolders = $homePath . 'wp-admin';
        $coreIncludesFolder = $homePath . 'wp-includes';

        if (is_array($dirs)) {
            foreach ($dirs as $item) {
                $path = $dir . '/' . $item['name'];

                if ($item['type'] == 'd') {
                    if ($structureChecking && !(stripos($path, $coreAdminFolders) === 0 || stripos($path, $coreIncludesFolder) === 0)) {

                        continue;
                    }

                    $help = $this->wpStoragePath;
                    $files = array_merge($files, $this->getAllPhpFilesInDirectory($path, $structureChecking));
                    $this->wpStoragePath = $help;
                } else {
                    if ($item['type'] === 'f') {
                        preg_match('/\.php$/', $item['name'], $matchExt);
                        if (!empty($matchExt)) {
                            $files[] = $path;
                        }
                    }
                }
            }
        }

        $this->wpStoragePath = null;

        return $files;
    }

	public function checkIsEnabledXmlRpc(): bool
	{
		$fileName = $this->xmlrpcFileDetect();
		if ($fileName) {
			$args = ['headers' => ['X-Bw-Nonce' => self::generateNonce()]];
			$response = wp_remote_post(site_url("/{$fileName}"), $args);

			return !is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200;
		}

		$settingsModel = new SettingsModel();
		$xmlRpcDbIsEnabled = $settingsModel->get_by_name($settingsModel->xml_rpc_enabled) ?? null;
		
		if ($xmlRpcDbIsEnabled) {
			$xmlRpcIsEnabled = (int) $xmlRpcDbIsEnabled->value_number;
		} else {
			return false;
		}
		
		if (in_array($xmlRpcIsEnabled, [SettingsModel::XML_RPC_UNKNOWN, SettingsModel::XML_RPC_ENABLED])) {
			$settingsModel->update_by_name($settingsModel->xml_rpc_enabled, [$settingsModel->settings_structure->value_number => SettingsModel::XML_RPC_DISABLED]);
		}

		return false;
	}

    public function xmlrpcFileDetect(): string
    {
        $defaultFileName = 'xmlrpc.php';
        $filePath = ABSPATH . $defaultFileName;
        $body = $this->wp_filesystem->get_contents($filePath);

        if ($this->xmlrpcFileStructureDetect($body)) {
            return $defaultFileName;
        }

        $files = $this->wp_filesystem->dirlist(ABSPATH);
        if ($files) {
            foreach ($files as $fileName => $data) {
                if ($data['type'] === 'f') {
                    $filePath = ABSPATH . $fileName;
                    $body = $this->wp_filesystem->get_contents($filePath);
                    if ($this->xmlrpcFileStructureDetect($body)) {
                        return $fileName;
                    }
                }
            }
        }

        return '';
    }

    public function xmlrpcFileStructureDetect(string $body): bool
    {
        preg_match_all('/xml-?rpc/i', $body, $match);
        if (count($match[0] ?? []) > 15) {
            preg_match("~^<\?php.*define\(.?'XMLRPC_REQUEST'.*require_once ABSPATH.*/class-wp-xmlrpc-server.php~s", $body, $match);
            if ($match) {
                return true;
            }
        }

        return false;
    }

    public function checkIsEnabledRss(): bool
    {
        $rssUrl = get_feed_link();
        $response = wp_remote_get($rssUrl);

        return !is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200;
    }

    public function getUnusedThemes(): array
    {
        $allThemes = wp_get_themes();
        $activeTheme = wp_get_theme();

        $unusedThemes = [];
        foreach ($allThemes as $themeSlug => $theme) {
            if ($themeSlug !== $activeTheme->get_stylesheet() && $themeSlug !== $activeTheme->get_template()) {
                $unusedThemes[$themeSlug] = $theme->get('Name');
            }
        }

        return $unusedThemes;
    }

    public function getUnusedPlugins(): array
    {
        $allPlugins = get_plugins();
        $activePlugins = get_option('active_plugins', []);

        $unusedPlugins = [];
        foreach ($allPlugins as $pluginFile => $pluginData) {
            if (!in_array($pluginFile, $activePlugins, true)) {
                $unusedPlugins[$pluginFile] = $pluginData['Name'];
            }
        }

        return $unusedPlugins;
    }

    public function deleteDirectory(string $directory): bool
    {
        if (!$this->wp_filesystem->is_dir($directory)) {
            return false;
        }

        $files = array_diff(scandir($directory), ['.', '..']);
        foreach ($files as $file) {
            $path = $directory . '/' . $file;
            if ($this->wp_filesystem->is_dir($path)) {
                $this->deleteDirectory($path);
            } else {
                $this->wp_filesystem->delete($path);
            }
        }

        return $this->wp_filesystem->rmdir($directory);
    }

    public function getDateTimeLastDatabaseUpdate(): string
    {
        $dbIpv4 = BinaryHelper::BINARY_DB_DIRECTORY . '/' . BinaryHelper::IPV4_FILE;
        $dbIpv6 = BinaryHelper::BINARY_DB_DIRECTORY . '/' . BinaryHelper::IPV6_FILE;

        $timezoneOffset = get_option('gmt_offset') * HOUR_IN_SECONDS;

        if ($this->wp_filesystem->exists($dbIpv4)) {
            $lastDatabaseUpdate = $this->wp_filesystem->mtime($dbIpv4);
            $lastDatabaseUpdate += $timezoneOffset;
            $lastDatabaseUpdate = date('Y-m-d H:i:s', $lastDatabaseUpdate);
        } else if ($this->wp_filesystem->exists($dbIpv6)) {
            $lastDatabaseUpdate = $this->wp_filesystem->mtime($dbIpv6);
            $lastDatabaseUpdate += $timezoneOffset;
            $lastDatabaseUpdate = date('Y-m-d H:i:s', $lastDatabaseUpdate);
        } else {
            $lastDatabaseUpdate = __('Database is missing', BW_PLUGIN_SLUG);
        }

        return $lastDatabaseUpdate;
    }

    public function getFirewallProtectionStatus(): int
    {
        $binaryHelper = BinaryHelper::getInstance();
        if ($binaryHelper->searchInPermanentList(BinaryHelper::S) === false) {
            return FirewallHelper::FIREWALL_PROTECTION_DISABLED;
        }

        $settingsModel = new SettingsModel();
        $firewallProtectionSettingsModules = $settingsModel->get_firewall_protection_settings_modules();

        $isEnabled = 0;
        $isDisabled = 0;

        foreach ($firewallProtectionSettingsModules as $module) {
            if ($module->value_number != 1) {
                $isDisabled++;
            } else {
                $isEnabled++;
            }
        }

        if ($isDisabled >= 1 && $isEnabled >= 1) {
            return FirewallHelper::FIREWALL_PROTECTION_PARTIALLY_ENABLED;
        }
        if ($isEnabled < 1) {
            return FirewallHelper::FIREWALL_PROTECTION_DISABLED;
        }

        return FirewallHelper::FIREWALL_PROTECTION_ENABLED;
    }

    public function getCurrentProtectionInfo(): array
    {
        $firewallProtectionStatus = $this->getFirewallProtectionStatus();
        $navManager = new NavManager();

        switch ($firewallProtectionStatus) {
            case FirewallHelper::FIREWALL_PROTECTION_PARTIALLY_ENABLED:
                $protectionStatusTitle = __('Protection partially enabled', BW_PLUGIN_SLUG);
                $protectionStatusIcon = 'partially-enabled-ic';
            break;
            case FirewallHelper::FIREWALL_PROTECTION_ENABLED:
                $protectionStatusTitle = __('Protection enabled', BW_PLUGIN_SLUG);
                $protectionStatusIcon = 'enabled-ic';
            break;
            default:
                $protectionStatusTitle = __('Protection disabled', BW_PLUGIN_SLUG);
                $protectionStatusIcon = 'disabled-ic';
            break;
        }
        return [
            'protection_status_title' => $protectionStatusTitle,
            'protection_status_icon' => $protectionStatusIcon,
            'link_settings' => [
                'needed' => $firewallProtectionStatus == FirewallHelper::FIREWALL_PROTECTION_PARTIALLY_ENABLED,
                'link' => $navManager->generate_url('firewall', 'settings'),
                'text' => __('Settings', BW_PLUGIN_SLUG),
            ],
            'button_enable' => [
                'needed' => $firewallProtectionStatus == FirewallHelper::FIREWALL_PROTECTION_DISABLED,
                'text' => __('Enable', BW_PLUGIN_SLUG),
            ],
        ];
    }

    public function validateLimitValue(string $value, int $min = 0, int $max = 100000): bool
    {
        if (!preg_match('/^\d+$/', $value)) {
            return false;
        }
        $intValue = (int)$value;
        return $intValue >= $min && $intValue <= $max;
    }

    public static function user_can_affect_site(): bool
    {
        if (!is_user_logged_in()) {
            return false;
        }

        $caps = [
            'switch_themes',
            'edit_themes',
            'edit_theme_options',
            'install_themes',
            'activate_plugins',
            'edit_plugins',
            'install_plugins',
            'edit_users',
            'edit_files',
            'manage_options',
            'moderate_comments',
            'manage_categories',
            'manage_links',
            'upload_files',
            'import',
            'unfiltered_html',
            'edit_posts',
            'edit_others_posts',
            'edit_published_posts',
            'publish_posts',
            'edit_pages',
            'publish_pages',
            'edit_others_pages',
            'edit_published_pages',
            'delete_pages',
            'delete_others_pages',
            'delete_published_pages',
            'delete_posts',
            'delete_others_posts',
            'delete_published_posts',
            'delete_private_posts',
            'edit_private_posts',
            'read_private_posts',
            'delete_private_pages',
            'edit_private_pages',
            'read_private_pages',
            'delete_users',
            'create_users',
            'unfiltered_upload',
            'edit_dashboard',
            'customize',
            'update_plugins',
            'delete_plugins',
            'update_themes',
            'update_core',
            'list_users',
            'remove_users',
            'add_users',
            'promote_users',
            'delete_themes',
            'export',
            'create_sites',
            'delete_sites',
            'manage_network',
            'manage_sites',
            'manage_network_users',
            'manage_network_themes',
            'manage_network_options',
            'manage_network_plugins',
            'upload_plugins',
            'upload_themes',
            'upgrade_network',
            'setup_network',
        ];

        foreach ($caps as $cap) {
            if (current_user_can($cap)) {
                return true;
            }
        }

        return false;
    }

}