<?php
namespace betawall\helpers;

use betawall\api\data\ActivationRequestData;
use betawall\api\requests\ActivationRequest;
use betawall\db\models\FirewallBlacklistModel;
use betawall\db\models\FirewallWhitelistModel;
use betawall\db\models\MalwareModel;
use betawall\db\models\SettingsModel;

class InitHelper
{

    public function on_activation()
    {
        //Init settings table
        $settingsModel = new SettingsModel();
        $settingsModel->settings_structure->prepare_create_table();

        //Init firewall statistics table
        $malwareModel = new MalwareModel();
        $malwareModel->malware_structure->prepare_create_table();

        //Init firewall blocked table
        $firewallBlacklistModel = new FirewallBlacklistModel();
        $firewallBlacklistModel->firewall_blacklist_structure->prepare_create_table();

        //Init firewall whitelist table
        $firewallWhitelistModel = new FirewallWhitelistModel();
        $firewallWhitelistModel->firewall_whitelist_structure->prepare_create_table();

        //Init secret key
        $secretKeyItem = $settingsModel->get_by_name($settingsModel->secret_key);
        $newSecretKey = MainHelper::generateSecretKey();
        if (!empty($secretKeyItem)) {
            $settingsModel->update_by_id($secretKeyItem->id, [
                $settingsModel->settings_structure->value_text => $newSecretKey
            ]);
        } else {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->secret_key,
                $settingsModel->settings_structure->value_text => $newSecretKey,
                $settingsModel->settings_structure->value_number => 0
            ]);
        }


        //Init unblock key
        $unblockKeyItem = $settingsModel->get_by_name($settingsModel->unblock_key);
        $newUnblockKey = MainHelper::generateSecretKey();
        if (!empty($unblockKeyItem)) {
            $settingsModel->update_by_id(
                $unblockKeyItem->id, [
                    $settingsModel->settings_structure->value_text => $newUnblockKey
                ]);
        } else {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->unblock_key,
                $settingsModel->settings_structure->value_text => $newUnblockKey,
                $settingsModel->settings_structure->value_number => 0
            ]);
        }


        //Init global error states
        $errorStates = $settingsModel->get_by_name($settingsModel->error_states);
        if (empty($errorStates)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->error_states,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::ERROR_STATE_NEED_ACTIVATION_MODAL
            ]);
        } else {
            $settingsModel->update_by_name($settingsModel->error_states, [
                $settingsModel->settings_structure->value_number => SettingsModel::ERROR_STATE_NEED_ACTIVATION_MODAL,
            ]);
        }


        //Init access bin files state
        $accessState = $settingsModel->get_by_name($settingsModel->access_state);
        if (empty($accessState)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->access_state,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::ACCESS_STATE_ALLOWED
            ]);
        } else {
            $settingsModel->update_by_name($settingsModel->access_state, [
                $settingsModel->settings_structure->value_number => SettingsModel::ACCESS_STATE_ALLOWED,
            ]);
        }


        //Firewall notfound module
        $firewallNotfoundEnabled = $settingsModel->get_by_name($settingsModel->firewall_notfound_enabled);
        if (empty($firewallNotfoundEnabled)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_notfound_enabled,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_NOTFOUND_DISABLED
            ]);
        }
        $firewallNotfound5mLimit = $settingsModel->get_by_name($settingsModel->firewall_notfound_5m_limit);
        if (empty($firewallNotfound5mLimit)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_notfound_5m_limit,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_NOTFOUND_5M_LIMIT_DEFAULT
            ]);
        }
        $firewallNotfound24hLimit = $settingsModel->get_by_name($settingsModel->firewall_notfound_24h_limit);
        if (empty($firewallNotfound24hLimit)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_notfound_24h_limit,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_NOTFOUND_24H_LIMIT_DEFAULT
            ]);
        }


        //Firewall failed login module
        $firewallFailedLoginEnabled = $settingsModel->get_by_name($settingsModel->firewall_failed_login_enabled);
        if (empty($firewallFailedLoginEnabled)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_failed_login_enabled,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_FAILED_LOGIN_DISABLED
            ]);
        }
        $firewallFailedLogin5mLimit = $settingsModel->get_by_name($settingsModel->firewall_failed_login_5m_limit);
        if (empty($firewallFailedLogin5mLimit)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_failed_login_5m_limit,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_FAILED_LOGIN_5M_LIMIT_DEFAULT
            ]);
        }
        $firewallFailedLogin24hLimit = $settingsModel->get_by_name($settingsModel->firewall_failed_login_24h_limit);
        if (empty($firewallFailedLogin24hLimit)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_failed_login_24h_limit,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_FAILED_LOGIN_24H_LIMIT_DEFAULT
            ]);
        }


        //Firewall SQL-injection module
        $firewallSqlInjectionEnabled = $settingsModel->get_by_name($settingsModel->firewall_sql_injection_enabled);
        if (empty($firewallSqlInjectionEnabled)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_sql_injection_enabled,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_SQL_INJECTION_DISABLED
            ]);
        }
        $firewallSqlInjection5mLimit = $settingsModel->get_by_name($settingsModel->firewall_sql_injection_5m_limit);
        if (empty($firewallSqlInjection5mLimit)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_sql_injection_5m_limit,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_SQL_INJECTION_5M_LIMIT_DEFAULT
            ]);
        }
        $firewallSqlInjection24hLimit = $settingsModel->get_by_name($settingsModel->firewall_sql_injection_24h_limit);
        if (empty($firewallSqlInjection24hLimit)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_sql_injection_24h_limit,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_SQL_INJECTION_24H_LIMIT_DEFAULT
            ]);
        }


        //Firewall code execution module
        $firewallCodeExecutionEnabled = $settingsModel->get_by_name($settingsModel->firewall_code_execution_enabled);
        if (empty($firewallCodeExecutionEnabled)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_code_execution_enabled,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_CODE_EXECUTION_DISABLED
            ]);
        }
        $firewallCodeExecution5mLimit = $settingsModel->get_by_name($settingsModel->firewall_code_execution_5m_limit);
        if (empty($firewallCodeExecution5mLimit)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_code_execution_5m_limit,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_CODE_EXECUTION_5M_LIMIT_DEFAULT
            ]);
        }
        $firewallCodeExecution24hLimit = $settingsModel->get_by_name($settingsModel->firewall_code_execution_24h_limit);
        if (empty($firewallCodeExecution24hLimit)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_code_execution_24h_limit,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_CODE_EXECUTION_24H_LIMIT_DEFAULT
            ]);
        }


        //Firewall xss module
        $firewallXssEnabled = $settingsModel->get_by_name($settingsModel->firewall_xss_enabled);
        if (empty($firewallXssEnabled)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_xss_enabled,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_XSS_DISABLED
            ]);
        }
        $firewallXss5mLimit = $settingsModel->get_by_name($settingsModel->firewall_xss_5m_limit);
        if (empty($firewallXss5mLimit)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_xss_5m_limit,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_XSS_5M_LIMIT_DEFAULT
            ]);
        }
        $firewallXss24hLimit = $settingsModel->get_by_name($settingsModel->firewall_xss_24h_limit);
        if (empty($firewallXss24hLimit)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_xss_24h_limit,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_XSS_24H_LIMIT_DEFAULT
            ]);
        }


        //Firewall rfi module
        $firewallRfiEnabled = $settingsModel->get_by_name($settingsModel->firewall_rfi_enabled);
        if (empty($firewallRfiEnabled)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_rfi_enabled,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_RFI_DISABLED
            ]);
        }
        $firewallRfi5mLimit = $settingsModel->get_by_name($settingsModel->firewall_rfi_5m_limit);
        if (empty($firewallRfi5mLimit)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_rfi_5m_limit,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_RFI_5M_LIMIT_DEFAULT
            ]);
        }
        $firewallRfi24hLimit = $settingsModel->get_by_name($settingsModel->firewall_rfi_24h_limit);
        if (empty($firewallRfi24hLimit)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_rfi_24h_limit,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_RFI_24H_LIMIT_DEFAULT
            ]);
        }

        $firewallRfiExternalUrlsEnabled = $settingsModel->get_by_name($settingsModel->firewall_rfi_external_urls_enabled);
        if (empty($firewallRfiExternalUrlsEnabled)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_rfi_external_urls_enabled,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_RFI_EXTERNAL_URLS_DISABLED
            ]);
        }


        //Firewall suspicious file uploads
        $firewallSuspiciousFileUploadsEnabled = $settingsModel->get_by_name($settingsModel->firewall_suspicious_file_uploads_enabled);
        if (empty($firewallSuspiciousFileUploadsEnabled)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_suspicious_file_uploads_enabled,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_SUSPICIOUS_FILE_UPLOADS_DISABLED
            ]);
        }
        $firewallSuspiciousFileUploads5mLimit = $settingsModel->get_by_name($settingsModel->firewall_suspicious_file_uploads_5m_limit);
        if (empty($firewallSuspiciousFileUploads5mLimit)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_suspicious_file_uploads_5m_limit,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_SUSPICIOUS_FILE_UPLOADS_5M_LIMIT_DEFAULT
            ]);
        }
        $firewallSuspiciousFileUploads24hLimit = $settingsModel->get_by_name($settingsModel->firewall_suspicious_file_uploads_24h_limit);
        if (empty($firewallSuspiciousFileUploads24hLimit)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->firewall_suspicious_file_uploads_24h_limit,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::FIREWALL_SUSPICIOUS_FILE_UPLOADS_24H_LIMIT_DEFAULT
            ]);
        }


        //General scan rest API
        $restApiEnabled = $settingsModel->get_by_name($settingsModel->rest_api_enabled);
        if (empty($restApiEnabled)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->rest_api_enabled,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::REST_API_UNKNOWN
            ]);
        }


        //General scan XML-RPC
        $xmlRpcEnabled = $settingsModel->get_by_name($settingsModel->xml_rpc_enabled);
        if (empty($xmlRpcEnabled)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->xml_rpc_enabled,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::XML_RPC_UNKNOWN
            ]);
        }


        //General scan RSS
        $rssEnabled = $settingsModel->get_by_name($settingsModel->rss_enabled);
        if (empty($rssEnabled)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->rss_enabled,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => SettingsModel::RSS_UNKNOWN
            ]);
        }

        //General scan red dot indicator
        $generalScanRedDot = $settingsModel->get_by_name($settingsModel->general_scan_red_dot);
        if (empty($generalScanRedDot)) {
            $settingsModel->add([
                $settingsModel->settings_structure->name => $settingsModel->general_scan_red_dot,
                $settingsModel->settings_structure->value_text => '',
                $settingsModel->settings_structure->value_number => 0
            ]);
        }

        //Allowed IPs
        $allowedIps = $settingsModel->get_by_name($settingsModel->allowed_ips_enabled);
        if (empty($allowedIps)) {
            $settingsModel->easyAdd($settingsModel->allowed_ips_enabled, '', SettingsModel::ALLOWED_IPS_DISABLED);
        }

        //RECAPTCHA
        $recaptcha = $settingsModel->get_by_name($settingsModel->recaptcha_enabled);
        if (empty($recaptcha)) {
            $settingsModel->easyAdd($settingsModel->recaptcha_enabled, '', SettingsModel::RECAPTCHA_DISABLED);
        }
        $recaptchaSiteKey = $settingsModel->get_by_name($settingsModel->recaptcha_site_key);
        if (empty($recaptchaSiteKey)) {
            $settingsModel->easyAdd($settingsModel->recaptcha_site_key);
        }
        $recaptchaSecretKey = $settingsModel->get_by_name($settingsModel->recaptcha_secret_key);
        if (empty($recaptchaSecretKey)) {
            $settingsModel->easyAdd($settingsModel->recaptcha_secret_key);
        }

        //Multi-user 2FA
        $g2fa = $settingsModel->get_by_name($settingsModel->g2fa_enabled);
        if (empty($g2fa)) {
            $settingsModel->easyAdd($settingsModel->g2fa_enabled, '', SettingsModel::G2FA_DISABLED);
        }

        //Enable statistics send
        if (!wp_next_scheduled(TasksHelper::BW_SCHEDULE_MALWARE_SEND)) {
            wp_schedule_event(current_time('timestamp') + 86400, TasksHelper::INTERVAL_DAILY, TasksHelper::BW_SCHEDULE_MALWARE_SEND);
        }

        //Enable db updates
        if (!wp_next_scheduled(TasksHelper::BW_SCHEDULE_MALWARE_GET_UPDATE)) {
            wp_schedule_event(current_time('timestamp') + 86400, TasksHelper::INTERVAL_DAILY, TasksHelper::BW_SCHEDULE_MALWARE_GET_UPDATE);
        }

        BinaryHelper::getInstance()->initSysFiles();

        $wp_filesystem = FileSystemHelper::get_filesystem();
        if ($wp_filesystem && !$wp_filesystem->is_dir(BW_BACKUPS_DIR)) {
            $wp_filesystem->mkdir(BW_BACKUPS_DIR, 0777);
        }
		
		if (defined('WP_CLI') && WP_CLI) {
			$this->activationProcess();
			
			$muHelper = new MuHelper();
			$muHelper->changeMuPluginMode(true);
		}
    }

    public function on_deactivation()
    {
        //Disable mu-plugins mode
        $muHelper = new MuHelper();
        $muHelper->changeMuPluginMode(false);

        //Disable statistics send
        wp_clear_scheduled_hook(TasksHelper::BW_SCHEDULE_MALWARE_SEND);

        //Disable db updates
        wp_clear_scheduled_hook(TasksHelper::BW_SCHEDULE_MALWARE_GET_UPDATE);
    }

    public function error_notice()
    {
        $settingsModel = new SettingsModel();

        $globalErrors = [
            SettingsModel::ERROR_STATE_ERROR_ACTIVATION => [
                'class' => 'notice-error',
                'message' => __('Activation Error: The website could not be activated in the system. Try reinstalling the plugin, and if the issue persists, please contact support for assistance.', BW_PLUGIN_SLUG),
                'button' => [
                    'show' => true,
                    'class' => 'button-primary bw-support-notice-btn',
                    'text' => __('Support', BW_PLUGIN_SLUG),
                    'href' => MainHelper::getSupportLink()
                ]
            ],
            SettingsModel::ERROR_STATE_WRONG_DOMAIN => [
                'class' => 'notice-error',
                'message' => __('Error: The domain does not match the one registered in the system. This may occur if the website\'s domain has been changed. Please contact support for further information and assistance.', BW_PLUGIN_SLUG),
                'button' => [
                    'show' => true,
                    'class' => 'button-primary bw-support-notice-btn',
                    'text' => __('Support', BW_PLUGIN_SLUG),
                    'href' => MainHelper::getSupportLink()
                ]
            ],
            SettingsModel::ERROR_STATE_NOT_VERIFIED => [
                'class' => 'notice-error',
                'message' => __('Verification Error: The website could not be verified in the system. Please contact support for further information and assistance.', BW_PLUGIN_SLUG),
                'button' => [
                    'show' => true,
                    'class' => 'button-primary bw-support-notice-btn',
                    'text' => __('Support', BW_PLUGIN_SLUG),
                    'href' => MainHelper::getSupportLink()
                ]
            ],
            SettingsModel::ERROR_STATE_DEACTIVATED => [
                'class' => 'notice-error',
                'message' => __('Error: The website has been deactivated in the system. Please contact support for further information and assistance.', BW_PLUGIN_SLUG),
                'button' => [
                    'show' => true,
                    'class' => 'button-primary bw-support-notice-btn',
                    'text' => __('Support', BW_PLUGIN_SLUG),
                    'href' => MainHelper::getSupportLink()
                ],
            ],
            SettingsModel::ERROR_STATE_NEED_ACTIVATION => [
                'class' => 'notice-error',
                'message' => __('The website requires activation in the system. Please click the "Activate" below to activate it.', BW_PLUGIN_SLUG),
                'button' => [
                    'show' => true,
                    'class' => 'button-primary bw-activation-plugin-btn',
                    'text' => __('Activate', BW_PLUGIN_SLUG)
                ],
            ],
            SettingsModel::ERROR_STATE_NEED_ACTIVATION_MODAL => [
                'class' => 'notice-error',
                'message' => __('The website requires activation in the system. Please click the "Activate" below to activate it.', BW_PLUGIN_SLUG),
                'button' => [
                    'show' => true,
                    'class' => 'button-primary bw-activation-plugin-btn',
                    'text' => __('Activate', BW_PLUGIN_SLUG)
                ],
            ]
        ];

        $globalErrorState = $settingsModel->get_by_name($settingsModel->error_states)->value_number ?? SettingsModel::ERROR_STATE_NO_ERRORS;
        $globalError = $globalErrors[$globalErrorState] ?? null;

        if ($globalError) {
            $noticeButton = $this->getNoticeButton($globalError['button']);
            $outputGlobalError = sprintf('
                    <div class="notice %s">
                        <div class="bw-notice-container">
                            <div class="bw-notice-icon"></div>
                            <div class="bw-notice-body">
                                <div class="bw-notice-title bw-black">%s</div>
                                <div class="bw-notice-info bw-black">%s</div>
                                <div class="bw-notice-button-container">
                                    <div class="bw-spinner spinner"></div>
                                    %s
                                </div>
                            </div>
                        </div>
                    </div>
                ',
                $globalError['class'],
                esc_html(BW_PLUGIN_NAME),
                esc_html($globalError['message']),
                $noticeButton
            );

            echo $outputGlobalError;
        }

        $accessErrors = [
            SettingsModel::ACCESS_STATE_DENIED => [
                'class' => 'notice-error',
                'message' => __('Error: The plugin cannot work with the required system files due to insufficient access permissions. Please ensure that all folders and files in the "wp-content/plugins/betawall/db/" directory have sufficient read and write permissions.', BW_PLUGIN_SLUG),
                'button' => [
                    'show' => true,
                    'class' => 'button-primary bw-support-notice-btn',
                    'text' => __('Support', BW_PLUGIN_SLUG),
                    'href' => MainHelper::getSupportLink()
                ],
            ],
            SettingsModel::ACCESS_FEATURE_DENIED => [
                'class' => 'notice-error',
                'message' => __('You are using the free version — functionality may be limited.', BW_PLUGIN_SLUG),
                'button' => [
                    'show' => true,
                    'class' => 'button-primary bw-upgrade-plugin-btn',
                    'text' => __('Upgrade', BW_PLUGIN_SLUG)
                ],
            ]
        ];

        $accessErrorState = $settingsModel->get_by_name($settingsModel->access_state)->value_number ?? SettingsModel::ACCESS_STATE_ALLOWED;
        $accessError = $accessErrors[$accessErrorState] ?? null;

        if ($accessError) {
            $noticeButton = $this->getNoticeButton($accessError['button']);
            $outputAccessError = sprintf('
                    <div class="notice %s">
                        <div class="bw-notice-container">
                            <div class="bw-notice-icon"></div>
                            <div class="bw-notice-body">
                                <div class="bw-notice-title bw-black">%s</div>
                                <div class="bw-notice-info bw-black">%s</div>
                                %s
                            </div>
                        </div>
                    </div>
                ',
                $accessError['class'],
                esc_html(BW_PLUGIN_NAME),
                esc_html($accessError['message']),
                $noticeButton
            );

            echo $outputAccessError;
        }

        $featureErrorState = BinaryHelper::getInstance()->searchInPermanentList(BinaryHelper::S) === false;
        $accessError = ($featureErrorState) ? $accessErrors[2] ?? null : null;
        if ($accessError) {
            $noticeButton = $this->getNoticeButton($accessError['button']);
            $outputAccessError = sprintf('
                    <div class="notice %s">
                        <div class="bw-notice-container">
                            <div class="bw-notice-icon"></div>
                            <div class="bw-notice-body">
                                <div class="bw-notice-title bw-black">%s</div>
                                <div class="bw-notice-info bw-black">%s</div>
                                %s
                            </div>
                        </div>
                    </div>
                ',
                $accessError['class'],
                esc_html(BW_PLUGIN_NAME),
                esc_html($accessError['message']),
                $noticeButton
            );

            echo $outputAccessError;
        }
    }

    private function getNoticeButton($buttonConfig)
    {
        if (empty($buttonConfig['show'])) {
            return '';
        }

        $class = isset($buttonConfig['class']) ? esc_attr($buttonConfig['class']) : '';
        $text = isset($buttonConfig['text']) ? esc_html($buttonConfig['text']) : '';
        $href = isset($buttonConfig['href']) ? esc_url($buttonConfig['href']) : '';

        if (!empty($href)) {
            return sprintf('<a href="%s" target="_blank" class="button %s">%s</a>', $href, $class, $text);
        }
        return sprintf('<button class="button %s">%s</button>', $class, $text);
    }

    public function init_widget()
    {
        $mainHelper = new MainHelper();

        $firewallProtectionStatus = $mainHelper->getFirewallProtectionStatus();
        switch ($firewallProtectionStatus) {
            case FirewallHelper::FIREWALL_PROTECTION_PARTIALLY_ENABLED:
                $protectionStatusTitle = __('Protection partially enabled', BW_PLUGIN_SLUG);
                $protectionStatusIcon = 'partially';
            break;
            case FirewallHelper::FIREWALL_PROTECTION_ENABLED:
                $protectionStatusTitle = __('Protection enabled', BW_PLUGIN_SLUG);
                $protectionStatusIcon = 'enabled';
            break;
            default:
                $protectionStatusTitle = __('Protection disabled', BW_PLUGIN_SLUG);
                $protectionStatusIcon = 'disabled';
            break;
        }

        $lastDatabaseUpdate = $mainHelper->getDateTimeLastDatabaseUpdate();

        $firewallBlacklistModel = new FirewallBlacklistModel();
        $last30dIpsBlockedCount = $firewallBlacklistModel->get_count_time_range(
            current_time('timestamp') - FirewallHelper::THIRTY_DAYS, current_time('timestamp')
        );
        $totalIpsBlockedCount = $firewallBlacklistModel->get_total_count();

        $firewallWhitelistModel = new FirewallWhitelistModel();
        $totalIpsWhitelistCount = $firewallWhitelistModel->get_total_count();

        $firewallUrl = esc_url(admin_url('admin.php?page=' . BW_PLUGIN_SLUG)) . '&c=firewall&a=index';

        $output = '
            <div class="bw-widget">
                <div class="bw-widget-protection-status ' . $protectionStatusIcon . '">' . $protectionStatusTitle . '</div>
                <div class="bw-widget-last-db-update bw-gray">' . __('Last database update', BW_PLUGIN_SLUG) . ': ' . $lastDatabaseUpdate . '</div>
                <div class="bw-widget-summary-info">
                    <div class="bw-widget-summary-col">
                        <div class="bw-widget-summary-col-head">' . __('Threat Summary', BW_PLUGIN_SLUG) . '</div>
                        <div class="bw-widget-summary-item">' . __('Last 30 days IPs Blocked', BW_PLUGIN_SLUG) . ' <a href="' . $firewallUrl . '&filter_date_range=30d' .  '">' . $last30dIpsBlockedCount . '</a></div>
                        <div class="bw-widget-summary-item">' . __('Total IPs Blocked', BW_PLUGIN_SLUG) . ' <a href="' . $firewallUrl . '">' . $totalIpsBlockedCount . '</a></div>
                    </div>
                    <div class="bw-widget-summary-col">
                        <div class="bw-widget-summary-col-head">' . __('Whitelist', BW_PLUGIN_SLUG) . '</div>
                        <div class="bw-widget-summary-item">'. __('Whitelisted IPs', BW_PLUGIN_SLUG) .' <a href="' . $firewallUrl . '">' . $totalIpsWhitelistCount . '</a></div>
                    </div>
                </div>
                <a href="' . $firewallUrl . '" class="button-primary">' . __('More details', BW_PLUGIN_SLUG) . '</a>
            </div>
        ';

        echo $output;
    }

    public function activationProcess()
    {
        $settingsModel = new SettingsModel();
        $siteUrl = MainHelper::getSiteUrl();
        $secretKeyItem = $settingsModel->get_by_name($settingsModel->secret_key);
        $unblockKeyItem = $settingsModel->get_by_name($settingsModel->unblock_key);
        $binaryHelper = BinaryHelper::getInstance();

        if (!empty($secretKeyItem)) {
            $unblockKeyValue = $unblockKeyItem->value_text ?? '';

            $pluginDir = parse_url(BW_PLUGIN_URL, PHP_URL_PATH);
            $pluginDirNormalized = trim($pluginDir, '/');

            $requestData = new ActivationRequestData($siteUrl, $secretKeyItem->value_text, $unblockKeyValue, BW_PLUGIN_VERSION, $pluginDirNormalized);
            $request = new ActivationRequest($requestData);
            $response = $request->sendRequest();

            if (!$response->success) {
                $settingsModel->update_by_name($settingsModel->error_states, [
                    $settingsModel->settings_structure->value_number => SettingsModel::ERROR_STATE_ERROR_ACTIVATION
                ]);
                $binaryHelper->setS();

                return false;
            }

            $settingsModel->update_by_name($settingsModel->error_states, [
                $settingsModel->settings_structure->value_number => SettingsModel::ERROR_STATE_NO_ERRORS
            ]);
            $tasksHelper = new TasksHelper();
            $result = $tasksHelper->malware_get_update();

            return $result->success && $result->s == BinaryHelper::S1;
        }

        $settingsModel->update_by_name($settingsModel->error_states, [
            $settingsModel->settings_structure->value_number => SettingsModel::ERROR_STATE_ERROR_ACTIVATION
        ]);
        $binaryHelper->setS();

        return false;
    }

}