<?php
namespace betawall\helpers;

use betawall\crud\FirewallWhitelistTableData;
use betawall\db\models\ClearModel;
use betawall\db\models\MalwareModel;
use betawall\db\models\SettingsModel;
use betawall\external\PHPGangsta_GoogleAuthenticator;
use WP_Error;

class FirewallHelper
{

    const FIVE_MINUTES = 300;
    const TWENTY_FOUR_HOURS = 86400;
    const THIRTY_DAYS = 86400 * 30;

    private static $instance;

    /** @var MuHelper */
    private $mu_helper;
    /** @var BinaryHelper */
    private $binary_helper;
    /** @var SettingsModel */
    private $settingsM;

    const FIREWALL_PROTECTION_DISABLED = 0;
    const FIREWALL_PROTECTION_ENABLED = 1;
    const FIREWALL_PROTECTION_PARTIALLY_ENABLED = 2;

    const CONTEXT_TYPE_GENERIC = 'generic';
    const CONTEXT_TYPE_GET = 'get';
    const CONTEXT_TYPE_POST = 'post';
    const CONTEXT_TYPE_HEADERS = 'headers';
    const CONTEXT_TYPE_FILES = 'files';

    const MAX_SCAN_BYTES_FILE = 4096;

    const UNBLOCK_URI = BW_PLUGIN_URL . 'scripts/unblock.php?key=';

    const WP_USER_ROLE_ADMIN = 'administrator';
    const WP_USER_ROLE_EDITOR = 'editor';
    const WP_USER_ROLE_AUTHOR = 'author';
    const WP_USER_ROLES = [
        self::WP_USER_ROLE_ADMIN,
        self::WP_USER_ROLE_EDITOR,
        self::WP_USER_ROLE_AUTHOR,
    ];
    private $wpUsers = [];

    private function __construct()
    {
        $this->mu_helper = new MuHelper();
        $this->binary_helper = BinaryHelper::getInstance();
        $this->settingsM = new SettingsModel();
    }

    public static function getInstance(): FirewallHelper
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function setLanguage()
    {
        $wp_filesystem = FileSystemHelper::get_filesystem();
        $locale = determine_locale();
        $languagesDir = BW_PLUGIN_PATH . 'languages/';
        $languageFile = $languagesDir . BW_PLUGIN_SLUG . "-$locale.mo";

        if ($wp_filesystem->exists($languageFile)) {
            load_textdomain(BW_PLUGIN_SLUG, $languageFile);
        }
    }

    public function track_init()
    {
        $this->setLanguage();

        if (defined('XMLRPC_REQUEST') && XMLRPC_REQUEST) {
            $settingsModel = new SettingsModel();
            $xmlRpcEnabled = $settingsModel->get_by_name($settingsModel->xml_rpc_enabled)->value_number ?? SettingsModel::XML_RPC_ENABLED;

            if ($xmlRpcEnabled == SettingsModel::XML_RPC_DISABLED) {
                status_header(403);
                exit('XML-RPC is disabled on this site.');
            }
        }

        $clientIp = MainHelper::getClientIp();
        $serverIp = gethostbyname(gethostname());

        if ($clientIp === $serverIp) {
            return;
        }

        $unblockPath = parse_url(self::UNBLOCK_URI, PHP_URL_PATH);
        if (strpos($_SERVER['REQUEST_URI'], $unblockPath) !== false) {
            return;
        }

        if ($this->binary_helper->isIpInRange($clientIp) !== false) {
            return;
        }

        if ($this->binary_helper->searchInFirewallList($clientIp, true) !== false) {
            return;
        }

        $isCurrentBanned = false;
        if (!$this->mu_helper->isEnabledMuPluginMode()) {
            $isCurrentBanned = $this->checkIsBanned($clientIp);
        }

        if (!MainHelper::user_can_affect_site()) {
            $this->checkRequest($clientIp, $isCurrentBanned);
        }
    }

    public function track_wp()
    {
        global $wp_query;

        if (is_feed()) {
            $settingsModel = new SettingsModel();
            $rssEnabled = $settingsModel->get_by_name($settingsModel->rss_enabled)->value_number ?? SettingsModel::RSS_ENABLED;
            if ($rssEnabled == SettingsModel::RSS_DISABLED) {
                wp_die(__('RSS feeds are disabled on this site.', BW_PLUGIN_SLUG), '', ['response' => 403]);
            }
        }

        $clientIp = MainHelper::getClientIp();
        $serverIp = gethostbyname(gethostname());

        if ($clientIp === $serverIp || MainHelper::user_can_affect_site()) {
            return;
        }

        $headers = getallheaders();
        $nonce = $headers['X-Bw-Nonce'] ?? '';
        if (!empty($nonce) && MainHelper::verifyNonce($nonce)) {
            return;
        }

        if ($wp_query->is_main_query() && $wp_query->is_404()) {
            if ($this->shouldIgnore404Request()) {
                return;
            }

            if ($this->binary_helper->isIpInRange($clientIp) !== false) {
                return;
            }

            if ($this->binary_helper->searchInFirewallList($clientIp, true) !== false) {
                return;
            }

            $settingsModel = new SettingsModel();
            $firewallNotFoundEnabled = ($this->binary_helper->searchInPermanentList(BinaryHelper::S) === false) ? null : $settingsModel->get_by_name($settingsModel->firewall_notfound_enabled)->value_number ?? null;

            if ($firewallNotFoundEnabled == SettingsModel::FIREWALL_NOTFOUND_ENABLED) {
                $malwareModel = new MalwareModel();
                $this->addStatItem($malwareModel, $clientIp, MalwareModel::TYPE_NOT_FOUND);

                $firewallNotfound5mLimit = $settingsModel->get_by_name($settingsModel->firewall_notfound_5m_limit)->value_number ?? null;
                $firewallNotfound24hLimit = $settingsModel->get_by_name($settingsModel->firewall_notfound_24h_limit)->value_number ?? null;

                if ($firewallNotfound5mLimit != null) {
                    $stats5MinutesCount = $malwareModel->get_count_requests_time_range(
                        $clientIp, MalwareModel::TYPE_NOT_FOUND,
                        current_time('timestamp') - self::FIVE_MINUTES,
                        current_time('timestamp')
                    );

                    if ($stats5MinutesCount >= $firewallNotfound5mLimit) {
                        $this->binary_helper->addToFirewallBlacklist($clientIp, MalwareModel::TYPE_NOT_FOUND);
                        $this->processBanIfNeeded($clientIp);
                    }
                }

                if ($firewallNotfound24hLimit != null) {
                    $stats24HoursCount = $malwareModel->get_count_requests_time_range(
                        $clientIp, MalwareModel::TYPE_NOT_FOUND,
                        current_time('timestamp') - self::TWENTY_FOUR_HOURS,
                        current_time('timestamp')
                    );

                    if ($stats24HoursCount >= $firewallNotfound24hLimit) {
                        $this->binary_helper->addToFirewallBlacklist($clientIp, MalwareModel::TYPE_NOT_FOUND);
                        $this->processBanIfNeeded($clientIp);
                    }
                }
            }
        }
    }

    private function shouldIgnore404Request(): bool
    {
        $uri = $_SERVER['REQUEST_URI'] ?? '';
        $accept = $_SERVER['HTTP_ACCEPT'] ?? '';
        $path = parse_url($uri, PHP_URL_PATH);
        $extension = pathinfo($path, PATHINFO_EXTENSION);

        $host = $_SERVER['HTTP_HOST'];
        $referer = $_SERVER['HTTP_REFERER'] ?? '';
        $fetchDest = $_SERVER['HTTP_SEC_FETCH_DEST'] ?? '';

        $excludedAcceptTypes = ['text/html', 'application/json'];
        $ignoreExtensions = [
            'php', 'html', 'htm', 'asp', 'aspx', 'jsp', 'cfm', 'pl',
            'env', 'ini', 'conf', 'cfg', 'yaml', 'yml', 'sql', 'db',
            'sqlite', 'mdb', 'accdb', 'sh', 'bash', 'bat', 'cmd', 'ps1',
            'pem', 'key', 'crt', 'csr', 'der', 'bak', 'backup', 'old',
            'orig', 'tmp', 'temp', 'txt', 'log', 'md', 'rst', 'readme',
            'license', 'changelog', 'json', 'xml', 'tpl', 'inc',
            'gitignore', 'gitattributes', 'editorconfig', 'htaccess', 'htpasswd'
        ];

        if ($fetchDest === 'iframe' || (!empty($referer) && stripos($referer, $host) === false)) {
            return true;
        }

        $basename = basename($path);
        if ($basename !== '' && $basename[0] === '.') {
            return false;
        }

        if (!empty($extension)) {
            $ext = strtolower($extension);
            if (!in_array($ext, $ignoreExtensions, true) && preg_match('/^[a-z0-9]{1,6}$/i', $ext)) {
                foreach ($excludedAcceptTypes as $type) {
                    if (stripos($accept, $type) !== false) {
                        return false;
                    }
                }
                return true;
            }
        }

        return false;
    }

    public function track_failed_login_request()
    {
        $clientIp = MainHelper::getClientIp();
        $serverIp = gethostbyname(gethostname());

        if ($clientIp === $serverIp) {
            return;
        }

        if (!MainHelper::user_can_affect_site()) {
            $settingsModel = new SettingsModel();
            $firewallFailedLoginEnabled = ($this->binary_helper->searchInPermanentList(BinaryHelper::S) === false) ? null : $settingsModel->get_by_name($settingsModel->firewall_failed_login_enabled)->value_number ?? null;

            if ($firewallFailedLoginEnabled == SettingsModel::FIREWALL_FAILED_LOGIN_ENABLED) {

                $malwareModel = new MalwareModel();
                $this->addStatItem($malwareModel, $clientIp, MalwareModel::TYPE_LOGIN);

                $firewallFailedLogin5mLimit = $settingsModel->get_by_name($settingsModel->firewall_failed_login_5m_limit)->value_number ?? null;
                $firewallFailedLogin24hLimit = $settingsModel->get_by_name($settingsModel->firewall_failed_login_24h_limit)->value_number ?? null;

                if ($firewallFailedLogin5mLimit != null) {
                    $stats5MinutesCount = $malwareModel->get_count_requests_time_range(
                        $clientIp, MalwareModel::TYPE_LOGIN,
                        current_time('timestamp') - self::FIVE_MINUTES,
                        current_time('timestamp')
                    );

                    if ($stats5MinutesCount >= $firewallFailedLogin5mLimit) {
                        $this->binary_helper->addToFirewallBlacklist($clientIp, MalwareModel::TYPE_LOGIN);
                        $this->processBanIfNeeded($clientIp);
                    }
                }

                if ($firewallFailedLogin24hLimit != null) {
                    $stats24HoursCount = $malwareModel->get_count_requests_time_range(
                        $clientIp, MalwareModel::TYPE_LOGIN,
                        current_time('timestamp') - self::TWENTY_FOUR_HOURS,
                        current_time('timestamp')
                    );

                    if ($stats24HoursCount >= $firewallFailedLogin24hLimit) {
                        $this->binary_helper->addToFirewallBlacklist($clientIp, MalwareModel::TYPE_LOGIN);
                        $this->processBanIfNeeded($clientIp);
                    }
                }
            }
        }
    }

    public function track_mu_request()
    {
        $clientIp = MainHelper::getClientIp();

        $unblockPath = parse_url(self::UNBLOCK_URI, PHP_URL_PATH);
        if (strpos($_SERVER['REQUEST_URI'], $unblockPath) !== false) {
            return;
        }

        if ($this->binary_helper->isIpInRange($clientIp) !== false) {
            return;
        }

        if ($this->binary_helper->searchInFirewallList($clientIp, true) !== false) {
            return;
        }

        $this->processBanIfNeeded($clientIp);
    }

    public function checkRequest(string $clientIp, bool $isCurrentBanned)
    {
        if ($this->binary_helper->searchInPermanentList(BinaryHelper::S) === false) {
            return;
        }

        $settingsModel = new SettingsModel();
        $firewallRfiExternalUrlsEnabled = ($this->binary_helper->searchInPermanentList(BinaryHelper::S) === false) ? SettingsModel::FIREWALL_RFI_EXTERNAL_URLS_DISABLED : $settingsModel->get_by_name($settingsModel->firewall_rfi_external_urls_enabled)->value_number ?? SettingsModel::FIREWALL_RFI_EXTERNAL_URLS_DISABLED;

        foreach ($_GET as $key => $value) {
            if (is_array($value)) {
                $value = json_encode($value, JSON_UNESCAPED_UNICODE);
            }
            if ($value && is_string($value) && $threatResult = $this->detectThreats($value, self::CONTEXT_TYPE_GET, $firewallRfiExternalUrlsEnabled)) {
                $this->hasThreat($threatResult, $clientIp, $isCurrentBanned);
                return;
            }
        }

        foreach ($_POST as $key => $value) {
            if (in_array($key, ['password', 'pass', 'pwd'], true)) {
                continue;
            }

            if (is_string($value)) {
                $decoded = json_decode($value, true);
                if (json_last_error() === JSON_ERROR_NONE) {
                    $value = $decoded;
                }
            }

            if (is_array($value)) {
                $value = json_encode($value, JSON_UNESCAPED_UNICODE);
            }

            if ($value && is_string($value) && $threatResult = $this->detectThreats($value, self::CONTEXT_TYPE_POST, $firewallRfiExternalUrlsEnabled)) {
                $this->hasThreat($threatResult, $clientIp, $isCurrentBanned);
                return;
            }
        }

        foreach (getallheaders() as $header => $value) {
            $excludedHeaders = [
                'accept', 'accept-encoding', 'accept-language', 'content-type', 'content-length',
                'connection', 'host', 'dnt', 'te', 'upgrade-insecure-requests'
            ];

            if (stripos($header, 'sec-') === 0 || in_array(strtolower($header), $excludedHeaders, true)) {
                continue;
            }
            if ($value && is_string($value) && $threatResult = $this->detectThreats($value, self::CONTEXT_TYPE_HEADERS, $firewallRfiExternalUrlsEnabled)) {
                $this->hasThreat($threatResult, $clientIp, $isCurrentBanned);
                return;
            }
        }

        if (!empty($_FILES)) {
            foreach ($_FILES as $file) {
                if ($threatResult = $this->detectFileUploadAttack($file, $firewallRfiExternalUrlsEnabled)) {
                    $this->hasThreat($threatResult, $clientIp, $isCurrentBanned);
                    return;
                }
            }
        }

        if ($isCurrentBanned) {
            self::accessForbiddenError();
        }
    }

    private function detectThreats(
        $input, $context = self::CONTEXT_TYPE_GENERIC,
        $firewallRfiExternalUrlsEnabled = SettingsModel::FIREWALL_RFI_EXTERNAL_URLS_DISABLED
    ) {
        $patterns = [
            MalwareModel::TYPE_SQL_INJECTION => [
                self::CONTEXT_TYPE_GENERIC => [
                    '/\bSELECT\s+\*\s+FROM\b/i',
                    '/\bSELECT\s+.+?\s+FROM\b/i',
                    '/\bINSERT\s+INTO\b/i',
                    '/\bUPDATE\s+\w+\s+SET\b/i',
                    '/\bDELETE\s+FROM\b/i',
                    '/\bDROP\s+TABLE\b/i',
                    '/\bALTER\s+TABLE\b/i',
                    '/\bTRUNCATE\s+TABLE\b/i',
                    '/\bUNION\s+(ALL\s+)?SELECT\b/i',
                    '/\bWHERE\s+\w+\s*(=|LIKE|IN)\s+/i',
                    '/\bSLEEP\s*\(\d+\)/i',
                    '/\bBENCHMARK\s*\(\d+,\s*\w+\)/i',
                    '/\bLENGTH\s*\(\w+\)/i',
                    '/\bHAVING\s+(COUNT|MAX|MIN|AVG|SUM)\s*\(/i',
                    '/\bGROUP\s+BY\s+\w+\s*(HAVING|ORDER|LIMIT|,|COUNT|SUM|AVG|MIN|MAX|\()/i',
                    '/\bOR\b\s+(\d+=\d+|\'?\w+\'?=\'?\w+\'?|\btrue\b)/i',
                    '/\bAND\b\s+(\d+=\d+|\'?\w+\'?=\'?\w+\'?|\btrue\b|\bSLEEP\s*\(\d+\)|\bBENCHMARK\s*\(\d+,\s*\w+\))/i',
                ],
            ],

            MalwareModel::TYPE_CODE_EXECUTION => [
				self::CONTEXT_TYPE_GENERIC => [
					'/<\?(php|=)?\b/i',
					'/\b(eval|exec|system|passthru|shell_exec|popen|proc_open|assert|create_function|call_user_func|call_user_func_array|preg_replace_callback|include|require|include_once|require_once)\b\s*\(/i',
					'/(%3C|<)\?(php|=)?/i',
					'/(base64_decode|gzuncompress|gzinflate|str_rot13|convert_uudecode|hex2bin)\s*\(/i',

					'/php:\/\/filter/i',
					'/convert\.(base64\-encode|base64\-decode|uudecode|quoted\-printable\-decode)/i',
					'/php:\/\/(input|output|memory|temp)/i',
					'/php:\/\/filter\/convert\.(base64\-encode|base64\-decode).*?\/resource=/i',
					'/data:\/\/text\/plain;base64,/i',

					'/file:\/\/\//i',
					'/file:\/\/(.*?\/)?etc\/passwd/i',
					'/file:\/\/.*?\.(conf|ini|env|php|sh|log|txt)/i',
					'/file_put_contents\s*\(\s*(php:\/\/|data:\/\/)/i',

					'/__import__\s*\(\s*[\'"]subprocess[\'"]\s*\)\s*\.\s*(run|call|Popen)/i',
					'/(?:^|[;|&`$]\s*)(?:perl|python|ruby|php)\s*-e\s+(?:[\'"][^\'"]+[\'"]|\S+)/i',

					'/powershell\s*-Command\s*["\'][^"\']*["\']/i',

					'/\(\s*\{\s*:;\s*\}\s*;\s*/i',
					'/\(\)\s*\{\s*:\s*;\s*\}\s*;\s*.*?/i',
					'/env\s+.*?\(\s*\{\s*:;\s*\}/i',
				],
            ],

            MalwareModel::TYPE_XSS => [
                self::CONTEXT_TYPE_GENERIC => [
                    '/<script\b[^>]*>(.*?)<\/script>/is',
                    '/(%3C|<)script\b[^>]*(%3E|>)(.*?)<(%2F|\/)script(%3E|>)/is',
                    '/<.*\b(on\w+)\s*=\s*[\'"]?[^>]+[\'"]?/i',
                    '/(javascript|vbscript):/i',
                    '/(%6A|j|%4A)(%61|a|%41)(%76|v|%56)(%61|a|%41)(%73|s|%53)(%63|c|%43)(%72|r|%52)(%69|i|%49)(%70|p|%50)(%74|t|%54):/ix',
                    '/expression\s*\(/i',
                    '/<img[^>]*src\s*=\s*[\'"]\s*data:image\/svg\+xml/i',
                    '/style\s*=\s*[\'"][^\'"]*(behavior|expression|javascript|vbscript|@import)/i',
                    '/data:\s*image\/svg\+xml\s*;base64\s*,/i',
                    '/<iframe\b[^>]*src\s*=\s*[\'"](javascript|data):[^>]+>/i',
                    '/document\.write\s*\(/i',
                    '/data:\s*(text|application)\/[a-z0-9+\-\.]+\s*;base64\s*,/i'
                ],
            ],

            MalwareModel::TYPE_RFI => [
                self::CONTEXT_TYPE_GENERIC => [
                    '/\b(include|require|include_once|require_once)\b\s*[^;]*?\b(h.{1,2}tp|f.{1,2}tp):\/\//i',
                ],
            ]
        ];

        if ($firewallRfiExternalUrlsEnabled == SettingsModel::FIREWALL_RFI_EXTERNAL_URLS_ENABLED) {
            $parsedUrl = parse_url(home_url());
            $siteDomain = preg_quote($parsedUrl['host'], '/');
            $siteDomainPattern = '([a-z0-9.-]*\.)?' . $siteDomain;

            $httpPatterns = [
                '/\b(http|https|ftp):\/\/(?!' . $siteDomainPattern . '\b)/i',
                '/\b(h.{1,2}tp|f.{1,2}tp):\/\/(?!' . $siteDomainPattern . '\b)/i',
                '/\b(http|https|ftp)(%3A|%3a)(%2F|%2f){2}(?!' . $siteDomainPattern . ')/i',
                '/\b(h.{1,2}tp|f.{1,2}tp)(%3A|%3a)(%2F|%2f){2}(?!' . $siteDomainPattern . ')/i'
            ];

            $patterns[MalwareModel::TYPE_RFI][self::CONTEXT_TYPE_GET] = $httpPatterns;
            $patterns[MalwareModel::TYPE_RFI][self::CONTEXT_TYPE_POST] = $httpPatterns;
        }

        $contextPatterns = [];
        foreach ($patterns as $attackType => $contexts) {
            $contextPatterns[$attackType] = array_merge(
                $contexts[self::CONTEXT_TYPE_GENERIC] ?? [],
                $contexts[$context] ?? []
            );
        }

        foreach ($contextPatterns as $threat => $threatPatterns) {
            foreach ($threatPatterns as $pattern) {
                if (preg_match($pattern, $input)) {
                    return [
                        'threat' => $threat,
                        'pattern' => $pattern,
                        'context' => $context,
                    ];
                }
            }
        }

        if (MainHelper::isBase64Encoded($input)) {
            $decodedInput = base64_decode($input, true);
            if ($decodedInput !== false) {
                return $this->detectThreats($decodedInput, $context);
            }
        }

        return false;
    }

    private function detectFileUploadAttack($file, $firewallRfiExternalUrlsEnabled = SettingsModel::FIREWALL_RFI_EXTERNAL_URLS_DISABLED)
    {
        if (empty($file['name']) || empty($file['tmp_name'])) {
            return false;
        }

        $fileName = $file['name'];
        $fileTmp = $file['tmp_name'];

        if (preg_match('/\.(php|phtml|php3|php4|php5|phps|htaccess|ini|env)$/i', $fileName)) {
            return [
                'threat' => MalwareModel::TYPE_SUSPICIOUS_FILE,
                'pattern' => '/\.(php|phtml|php3|php4|php5|phps|htaccess|ini|env)$/i',
                'context' => self::CONTEXT_TYPE_FILES
            ];
        }

        if (preg_match('/\.(php\d?|phtml)\./i', $fileName)) {
            return [
                'threat' => MalwareModel::TYPE_SUSPICIOUS_FILE,
                'pattern' => '/\.(php\d?|phtml)\./i',
                'context' => self::CONTEXT_TYPE_FILES
            ];
        }

        $mimeType = mime_content_type($fileTmp);
        if ($mimeType !== false && in_array($mimeType, ['application/x-php', 'text/x-php'])) {
            return [
                'threat' => MalwareModel::TYPE_SUSPICIOUS_FILE,
                'pattern' => 'Mime content type application/x-php or text/x-php',
                'context' => self::CONTEXT_TYPE_FILES
            ];
        }

        $fileContent = file_get_contents($fileTmp, false, null, 0, self::MAX_SCAN_BYTES_FILE);
        if ($fileContent !== false) {
            $threat = $this->detectThreats($fileContent, self::CONTEXT_TYPE_FILES, $firewallRfiExternalUrlsEnabled);
            if (is_array($threat)) {
                return $threat;
            }
        }
        return false;
    }

    private function hasThreat(array $threatResult, string $clientIp, bool $isCurrentBanned)
    {
        if ($isCurrentBanned) {
            $permanentlyLogsSaveTypes = [
                MalwareModel::TYPE_SQL_INJECTION,
                MalwareModel::TYPE_CODE_EXECUTION,
                MalwareModel::TYPE_RFI,
                MalwareModel::TYPE_SUSPICIOUS_FILE
            ];

            if (in_array($threatResult['threat'], $permanentlyLogsSaveTypes)) {
                $malwareModel = new MalwareModel();
                $this->addStatItem($malwareModel, $clientIp, $threatResult['threat'], $threatResult);
            }

            self::accessForbiddenError();
        } else {
            $settingsModel = new SettingsModel();

            $firewallThreat5mLimit = null;
            $firewallThreat24hLimit = null;

            $isEnabledFirewallModule = false;

            switch ($threatResult['threat']) {
                case MalwareModel::TYPE_SQL_INJECTION:
                    $firewallSqlInjectionEnabled = ($this->binary_helper->searchInPermanentList(BinaryHelper::S) === false) ? null : $settingsModel->get_by_name($settingsModel->firewall_sql_injection_enabled)->value_number ?? null;
                    $isEnabledFirewallModule = $firewallSqlInjectionEnabled == SettingsModel::FIREWALL_SQL_INJECTION_ENABLED;
                    if ($isEnabledFirewallModule) {
                        $firewallThreat5mLimit = $settingsModel->get_by_name($settingsModel->firewall_sql_injection_5m_limit)->value_number ?? null;
                        $firewallThreat24hLimit = $settingsModel->get_by_name($settingsModel->firewall_sql_injection_24h_limit)->value_number ?? null;
                    }
                break;
                case MalwareModel::TYPE_CODE_EXECUTION:
                    $firewallCodeExecutionEnabled = ($this->binary_helper->searchInPermanentList(BinaryHelper::S) === false) ? null : $settingsModel->get_by_name($settingsModel->firewall_code_execution_enabled)->value_number ?? null;
                    $isEnabledFirewallModule = $firewallCodeExecutionEnabled == SettingsModel::FIREWALL_CODE_EXECUTION_ENABLED;
                    if ($isEnabledFirewallModule) {
                        $firewallThreat5mLimit = $settingsModel->get_by_name($settingsModel->firewall_code_execution_5m_limit)->value_number ?? null;
                        $firewallThreat24hLimit = $settingsModel->get_by_name($settingsModel->firewall_code_execution_24h_limit)->value_number ?? null;
                    }
                break;
                case MalwareModel::TYPE_XSS:
                    $firewallXssEnabled = ($this->binary_helper->searchInPermanentList(BinaryHelper::S) === false) ? null : $settingsModel->get_by_name($settingsModel->firewall_xss_enabled)->value_number ?? null;
                    $isEnabledFirewallModule = $firewallXssEnabled == SettingsModel::FIREWALL_XSS_ENABLED;
                    if ($isEnabledFirewallModule) {
                        $firewallThreat5mLimit = $settingsModel->get_by_name($settingsModel->firewall_xss_5m_limit)->value_number ?? null;
                        $firewallThreat24hLimit = $settingsModel->get_by_name($settingsModel->firewall_xss_24h_limit)->value_number ?? null;
                    }
                break;
                case MalwareModel::TYPE_RFI:
                    $firewallRfiEnabled = ($this->binary_helper->searchInPermanentList(BinaryHelper::S) === false) ? null : $settingsModel->get_by_name($settingsModel->firewall_rfi_enabled)->value_number ?? null;
                    $isEnabledFirewallModule = $firewallRfiEnabled == SettingsModel::FIREWALL_RFI_ENABLED;
                    if ($isEnabledFirewallModule) {
                        $firewallThreat5mLimit = $settingsModel->get_by_name($settingsModel->firewall_rfi_5m_limit)->value_number ?? null;
                        $firewallThreat24hLimit = $settingsModel->get_by_name($settingsModel->firewall_rfi_24h_limit)->value_number ?? null;
                    }
                break;
                case MalwareModel::TYPE_SUSPICIOUS_FILE:
                    $firewallSuspiciousFileUploadsEnable = ($this->binary_helper->searchInPermanentList(BinaryHelper::S) === false) ? null : $settingsModel->get_by_name($settingsModel->firewall_suspicious_file_uploads_enabled)->value_number ?? null;
                    $isEnabledFirewallModule = $firewallSuspiciousFileUploadsEnable == SettingsModel::FIREWALL_SUSPICIOUS_FILE_UPLOADS_ENABLED;
                    if ($isEnabledFirewallModule) {
                        $firewallThreat5mLimit = $settingsModel->get_by_name($settingsModel->firewall_suspicious_file_uploads_5m_limit)->value_number ?? null;
                        $firewallThreat24hLimit = $settingsModel->get_by_name($settingsModel->firewall_suspicious_file_uploads_24h_limit)->value_number ?? null;
                    }
                break;
            }

            if ($isEnabledFirewallModule) {
                $malwareModel = new MalwareModel();
                $this->addStatItem($malwareModel, $clientIp, $threatResult['threat'], $threatResult);

                if ($firewallThreat5mLimit != null) {
                    $stats5MinutesCount = $malwareModel->get_count_requests_time_range(
                        $clientIp, $threatResult['threat'],
                        current_time('timestamp') - self::FIVE_MINUTES,
                        current_time('timestamp')
                    );

                    if ($stats5MinutesCount >= $firewallThreat5mLimit) {
                        $this->binary_helper->addToFirewallBlacklist($clientIp, $threatResult['threat']);
                        $this->processBanIfNeeded($clientIp);
                    }
                }

                if ($firewallThreat24hLimit != null) {
                    $stats24HoursCount = $malwareModel->get_count_requests_time_range(
                        $clientIp, $threatResult['threat'],
                        current_time('timestamp') - self::TWENTY_FOUR_HOURS,
                        current_time('timestamp')
                    );

                    if ($stats24HoursCount >= $firewallThreat24hLimit) {
                        $this->binary_helper->addToFirewallBlacklist($clientIp, $threatResult['threat']);
                        $this->processBanIfNeeded($clientIp);
                    }
                }
            }
        }
    }

    private function addStatItem(MalwareModel $malwareModel, string $ip, int $requestType, array $threatResult = []) {
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Undefined';
        $logs = $this->getLogsData($threatResult);

        $malwareModel->add([
            $malwareModel->malware_structure->type => $requestType,
            $malwareModel->malware_structure->ip => $ip,
            $malwareModel->malware_structure->user_agent => $userAgent,
            $malwareModel->malware_structure->logs => $logs,
            $malwareModel->malware_structure->is_sent => MalwareModel::STATUS_NOT_SENT,
        ]);
    }

    private function getLogsData(array $threatResult = []) {
        $logs = [
            'REQUEST_METHOD' => $_SERVER['REQUEST_METHOD'] ?? 'Undefined',
            'REQUEST_URI' => $_SERVER['REQUEST_URI'] ?? 'Undefined',
            'HTTP_REFERER' => $_SERVER['HTTP_REFERER'] ?? 'Undefined',
            'QUERY_STRING' => $_SERVER['QUERY_STRING'] ?? 'Undefined',
            'REQUEST' => json_encode($_REQUEST) ?? 'Undefined',
        ];

        if (!empty($threatResult)) {
            $logs['CONTEXT'] = $threatResult['context'] ?? 'Undefined';
            $logs['PATTERN'] = $threatResult['pattern'] ?? 'Undefined';

			if (($threatResult['context'] ?? '') === self::CONTEXT_TYPE_HEADERS) {
				$headers = getallheaders();
				$logs['HEADERS'] = json_encode($headers) ?? 'Undefined';
			}
        }

        return json_encode($logs);
    }

    private function processBanIfNeeded(string $clientIp)
    {
        if ($this->binary_helper->searchInPermanentList(BinaryHelper::S) !== false) {
            if ($this->binary_helper->searchInPermanentList($clientIp) !== false) {
                self::accessForbiddenError();
            }

            if ($this->binary_helper->searchInFirewallList($clientIp) !== false) {
                self::accessForbiddenError();
            }
        }
    }

    private function checkIsBanned(string $clientIp)
    {
        if ($this->binary_helper->searchInPermanentList(BinaryHelper::S) === false) {
            return false;
        }

        if ($this->binary_helper->searchInPermanentList($clientIp) !== false) {
            return true;
        }

        if ($this->binary_helper->searchInFirewallList($clientIp) !== false) {
            return true;
        }

        return false;
    }

    public static function accessForbiddenError()
    {
        http_response_code(403);
        exit;
    }

    public static function internalRequestError()
    {
        http_response_code(400);
        exit;
    }

    // Google reCAPTCHA - start
    public function enqueue_recaptcha_script() // connecting Google reCAPTCHA API and JS scripts, only when it is enabled
    {
        $recaptchaState = ($this->binary_helper->searchInPermanentList(BinaryHelper::S) === false) ? SettingsModel::RECAPTCHA_DISABLED : $this->settingsM->get_by_name($this->settingsM->recaptcha_enabled)->value_number ?? SettingsModel::RECAPTCHA_DISABLED;

        if (!$recaptchaState) {
            return; // if it disabled, then don`t load scripts
        }

        if (strpos($_SERVER['REQUEST_URI'], 'wp-login.php') !== false || is_admin()) {
            $siteKey = $this->settingsM->get_by_name($this->settingsM->recaptcha_site_key)->value_text ?? false;

            if ($siteKey) {
                echo '<meta name="recaptcha-site-key" content="' . esc_attr($siteKey) . '">';
                wp_enqueue_script(
                    'recaptcha-js',
                    'https://www.google.com/recaptcha/api.js?render=' . esc_attr($siteKey),
                    [],
                    null,
                    true
                );
                wp_enqueue_script(
                    'custom-recaptcha-js',
                    plugins_url('../../assets/js/external/recaptcha.js', __FILE__),
                    ['recaptcha-js'],
                    null,
                    true
                );
            }
        }
    }

    public function verify_recaptcha_on_login($user, $username, $password)
    {
        $recaptchaState = ($this->binary_helper->searchInPermanentList(BinaryHelper::S) === false) ? SettingsModel::RECAPTCHA_DISABLED : $this->settingsM->get_by_name($this->settingsM->recaptcha_enabled)->value_number ?? SettingsModel::RECAPTCHA_DISABLED;

        if (!$recaptchaState) {
            return $user; // if reCAPTCHA is disabled, just skip the check
        }

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {

            // checking for request from 2FA login form
            if (($_POST['action'] ?? '') === 'mu2fa_check') {
                $g2fa = ($this->binary_helper->searchInPermanentList(BinaryHelper::S) === false) ? SettingsModel::G2FA_DISABLED : $this->settingsM->get_by_name($this->settingsM->g2fa_enabled)->value_number ?? SettingsModel::G2FA_DISABLED;
                if ($g2fa) {
                    return $user;
                }
            }

            $secretKey = $this->settingsM->get_by_name($this->settingsM->recaptcha_secret_key)->value_text ?? false;
            $response = $_POST['g-recaptcha-response'] ?? '';
            $remoteip = MainHelper::getClientIp();

            if (empty($secretKey) || empty($response)) {
                return new WP_Error('recaptcha_error', __('<strong>Error:</strong> reCAPTCHA did`t found.'));
            }

            $verifyResponse = wp_remote_post('https://www.google.com/recaptcha/api/siteverify', [
                'body' => [
                    'secret'   => $secretKey,
                    'response' => $response,
                    'remoteip' => $remoteip
                ]
            ]);

            if (is_wp_error($verifyResponse)) {
                return new WP_Error('recaptcha_error', __('<strong>Error:</strong> Failed reCAPTCHA checking.'));
            }

            $captchaSuccess = json_decode(wp_remote_retrieve_body($verifyResponse));

            if (!isset($captchaSuccess->success) || !$captchaSuccess->success) {
                return new WP_Error('recaptcha_error', __('<strong>Error:</strong> reCAPTCHA did`t pass.'));
            }

            $score = $captchaSuccess->score ?? 0;

            if ($score <= 0.3) {
                return new WP_Error('recaptcha_error', __('<strong>Error:</strong> reCAPTCHA detected You as a bot.'));
            }
        }

        return $user;
    }

    public function add_recaptcha_to_login_form()
    {
        $recaptchaState = ($this->binary_helper->searchInPermanentList(BinaryHelper::S) === false) ? SettingsModel::RECAPTCHA_DISABLED : $this->settingsM->get_by_name($this->settingsM->recaptcha_enabled)->value_number ?? SettingsModel::RECAPTCHA_DISABLED;

        if ($recaptchaState) {
            $siteKey = $this->settingsM->get_by_name($this->settingsM->recaptcha_site_key)->value_text ?? false;
            if (!empty($siteKey)) {
                echo '<meta name="recaptcha-site-key" content="' . esc_attr($siteKey) . '">';
            }
        }
    }
    // Google reCAPTCHA - end

    // Multi-user 2FA - start
    public function mu2fa_check_callback()
    {
        $settingsModel = new SettingsModel();
        $g2fa = ($this->binary_helper->searchInPermanentList(BinaryHelper::S) === false) ? SettingsModel::G2FA_DISABLED : $settingsModel->get_by_name($settingsModel->g2fa_enabled)->value_number ?? SettingsModel::G2FA_DISABLED;
        if (!$g2fa) {
            wp_send_json_success(['2fa_required' => false]);
        }

        remove_filter('authenticate', [$this, 'mu2fa_authenticate_user'], 30);

        if (empty($_POST['username']) || empty($_POST['password'])) {
            wp_send_json_error(['message' => 'Введите логин и пароль.']);
        }
        $username = sanitize_text_field($_POST['username']);
        $password = $_POST['password'];
        $user = wp_authenticate($username, $password);
        if (is_wp_error($user)) {
            wp_send_json_error(['message' => $user->get_error_message()]);
        }
        // Если для пользователя включена 2FA – сообщаем об этом
        if (get_user_meta($user->ID, 'g2fa_user_enabled', true)) {
            wp_send_json_success(['2fa_required' => true]);
        } else {
            wp_send_json_success(['2fa_required' => false]);
        }
    }

    public function mu2fa_authenticate_user($user, $username, $password)
    {
        if (is_wp_error($user) || !$user) {
            return $user;
        }

        $settingsModel = new SettingsModel();
        $g2fa = ($this->binary_helper->searchInPermanentList(BinaryHelper::S) === false) ? SettingsModel::G2FA_DISABLED : $settingsModel->get_by_name($settingsModel->g2fa_enabled)->value_number ?? SettingsModel::G2FA_DISABLED;
        if (!$g2fa) {
            return $user;
        }

        if (!get_user_meta($user->ID, 'g2fa_user_enabled', true)) {
            return $user;
        }

        $code = isset($_POST['mu2fa_code']) ? sanitize_text_field($_POST['mu2fa_code']) : '';

        if (empty($code)) {
            wp_logout();
            wp_redirect(add_query_arg('login', '2fa_empty', wp_login_url()));
            exit;
        }

        $user_secret = get_user_meta($user->ID, 'g2fa_user_secret', true);
        $ga = new PHPGangsta_GoogleAuthenticator();
        if (!$ga->verifyCode($user_secret, $code, 2)) {
            wp_logout();
            wp_redirect(add_query_arg('login', '2fa_invalid', wp_login_url()));
            exit;
        }

        return $user;
    }

    public function mu2fa_login_message($message)
    {
        if ( isset( $_GET['login'] ) ) {
            if ( $_GET['login'] == '2fa_empty' ) {
                $message .= '<div class="notice notice-error">' . __('<p><strong>Error</strong>: Enter the 2FA code.</p>', BW_PLUGIN_SLUG) . '</div>';
            } elseif ( $_GET['login'] == '2fa_invalid' ) {
                $message .= '<div class="notice notice-error">' . __('<p><strong>Error</strong>: Invalid 2FA code.</p>', BW_PLUGIN_SLUG) . '</div>';
            }
        }

        return $message;
    }

    public function mu2fa_enqueue_login_script()
    {
        //    allow to load actual js and css, for individual privileged hosts
        $paramTimestamp = BW_PLUGIN_VERSION;
        if (defined('BW_PLUGIN_ALLOWED_HOSTS')) {
            $host = parse_url(home_url(), PHP_URL_HOST);
            $paramTimestamp = in_array($host, BW_PLUGIN_ALLOWED_HOSTS) ? time() : BW_PLUGIN_VERSION;
        }

        wp_enqueue_script(
            'mu2fa-login',
            plugins_url('../../assets/js/external/mu2fa-login.js', __FILE__),
            ['jquery'],
            $paramTimestamp,
            true
        );
        wp_localize_script('mu2fa-login', 'mu2fa_ajax', [
            'ajax_url' => admin_url('admin-ajax.php')
        ]);
    }

    public function save_g2fa_changes(bool $status) :array
    {
        if ($status) {
            $result = $this->settingsM->update_by_name($this->settingsM->g2fa_enabled, [
                $this->settingsM->settings_structure->value_number => SettingsModel::G2FA_ENABLED,
            ]);
            $errorMessage = __('Could not enable g2fa', BW_PLUGIN_SLUG);
        } else {
            $result = $this->settingsM->update_by_name($this->settingsM->g2fa_enabled, [
                $this->settingsM->settings_structure->value_number => SettingsModel::G2FA_DISABLED,
            ]);
            $errorMessage = __('Could not disable g2fa', BW_PLUGIN_SLUG);
        }

        if ($result !== false) {
            $response = [
                'result' => 'successful',
            ];
        } else {
            $response = ['result' => 'error', 'message' => $errorMessage];
        }

        return $response;
    }

    public function save_g2fa_user_changes($userId, $value)
    {
        $model = new ClearModel();
        $table = 'usermeta';

        if ($value) {
            $fields = [
                'user_id' => $userId,
                'meta_key' => 'g2fa_user_enabled',
                'meta_value' => $value,
            ];
            $model->insert_into_custom_table($table, $fields);

            $ga = new PHPGangsta_GoogleAuthenticator();
            $secret = $ga->createSecret();
            $fields = [
                'user_id' => $userId,
                'meta_key' => 'g2fa_user_secret',
                'meta_value' => $secret,
            ];
            $model->insert_into_custom_table($table, $fields);
            $user = new \WP_User($userId);
            $ga = new PHPGangsta_GoogleAuthenticator();
            $qrUrl = $ga->getQRCodeGoogleUrl($user->user_login, $secret, get_bloginfo('name'));

            return [
                'secret_key' => $secret,
                'qr_url' => $qrUrl,
            ];
        } else {
            $field = 'user_id';
            $ids = [$userId];
            $where = [
                ['meta_key' => 'g2fa_user_enabled'],
                ['meta_key' => 'g2fa_user_secret'],
            ];

            $model->delete_custom_data_multiple_ids($table, $field, $ids, $where);

            return [];
        }
    }

    public function g2faUsersListHtmlBlock(int $g2faIsEnabled, string $userRole = self::WP_USER_ROLE_ADMIN) :string
    {
        $html = '';
        $class = !$g2faIsEnabled ? ' bw-state' : '';

        $blockExample =
        '<div class="bw-2fa-container js-bw-2fa-user js-bw-2fa-rows-with-data' . $class .'" data-role=":userRoleTitle:">
            <div class="bw-2fa-user">
                <div class="bw-2fa-block-name">
                    <div class="bw-2fa-name">:login:</div>
                    <div class="bw-2fa-role">:userRole:</div>
                </div>
                <div class="bw-2fa-block">
                    <div class=":statusClass:">:2faStatus:</div>
                    <div class="bw-2fa-edit js-bw-show-user-details" data-id=":userId:">' . __('Edit', BW_PLUGIN_SLUG) . '</div>
                </div>
            </div>
            <div class="bw-2fa-enable-2fa bw-state js-bw-user-details-block js-bw-user-details-block-:userId:">
                <label>
                    <input class="bw-firewall-settings-module-checkbox js-bw-2fa-user-id" type="checkbox" name="g2fa_user[:userId:]" data-id=":userId:" value=":val:" :checked:/>
                    ' . __('Enable 2FA for this user', BW_PLUGIN_SLUG) . '
                </label>
                :userHas2faBlockExample
             </div>
        </div>';

        $userHas2faBlockExample =
            '<div class="bw-2fa-secret-key">
                <label>' . __('Secret key', BW_PLUGIN_SLUG) . '</label>
                <input type="text" value=":user_secret" readonly="">
            </div>
            <div class="bw-2fa-qr-code">
                <label>QR-code</label>
                <img class="lazy" data-original=":qrUrl" alt="QR Code">
            </div>';

        $this->cachingWpUsers($userRole);
        foreach ($this->wpUsers as $user) {
            $block = $blockExample;
            $has2FaBlock = $userHas2faBlockExample;
            $userId = $user['id'];
            $user2fa = (int) $user['g2fa_user_enabled'];
            $secret = $user['g2fa_user_secret'];
            $userRoleTitle = __(esc_attr($user['role']), BW_PLUGIN_SLUG);

            $userRole = __($userRoleTitle, BW_PLUGIN_SLUG);

            if ($secret) {
                $ga    = new PHPGangsta_GoogleAuthenticator();
                $qrUrl = $ga->getQRCodeGoogleUrl($user['user_login'], $secret, get_bloginfo('name'));
            }

            if (esc_attr($user2fa)) {
                $state = __('2FA enabled', BW_PLUGIN_SLUG);
            } else {
                $state = __('2FA disabled', BW_PLUGIN_SLUG);
            }

            $checked = $user2fa ? 'checked="checked"' : '';
            $statusClass = $user2fa ? 'bw-2fa-status' : 'bw-2fa-status-disabled';

            $block = str_replace(
                [':login:',          ':2faStatus:', ':val:',  ':userRole:', ':userRoleTitle:', ':userId:', ':checked:',  ':statusClass:'],
                [$user['display_name'], $state,      $user2fa, $userRole,    $userRoleTitle,    $userId,    $checked,    $statusClass],
                $block
            );

            if ($user2fa) {
                $has2FaBlock = str_replace(':user_secret', $secret, $has2FaBlock);

                $block = str_replace([':userHas2faBlockExample', ':qrUrl'], [$has2FaBlock, $qrUrl], $block);
            } else {
                $block = str_replace(':userHas2faBlockExample', '', $block);
            }

            $html .= $block;
        }

        return '<div class="js-bw-2fa-user-container" style="display: none">' . $html . '</div>';
    }

    public function g2faRolesSelectHtmlBlock() :string
    {
        $rolesArray = $this->wpUserRoles();

        $optionsHtml = '';
        foreach ($rolesArray as $k => $roleTitle) {
            $selected = $k === 0 ? ' selected' : '';
            $optionsHtml .= '<option value="' . $roleTitle . '"' . $selected . '>' . __($roleTitle, BW_PLUGIN_SLUG) . '</option>';
        }

        $optionsHtml .= '<option value="another">' . __('All another', BW_PLUGIN_SLUG) . '</option>';

        return
            '<div class="bw-2fa-block-action js-bw-2fa-rows-with-data">' . __('Users', BW_PLUGIN_SLUG) . '
                <select class="js-bw-2fa-roles-select" name="user_role">
                    ' . $optionsHtml . '
                </select>
            </div>';
    }

    public function wpUserRoles() :array
    {
        $model = new ClearModel();

        $roles = "'" . implode("','", self::WP_USER_ROLES) . "'";
        $userRoles = $model->get_custom_data_from_tables_clear(
            "SELECT REPLACE(REGEXP_SUBSTR(um.meta_value, '\"(.*)\"'), '\"', '') AS role
            FROM {db_prefix}users u
            JOIN {db_prefix}usermeta um ON um.user_id = u.id AND um.meta_key = '{db_prefix}capabilities'
            GROUP BY role
            HAVING role IN ({$roles})"
        );

        $rolesArray = [];
        foreach ($userRoles as $user) {
            $rolesArray[$user['role']] = $user['role'];
        }
        asort($rolesArray);

        return array_values($rolesArray);
    }

    public function cachingWpUsers(string $userRole = self::WP_USER_ROLE_ADMIN)
    {
        $this->wpUsers = [];

        $userRole = sanitize_text_field($userRole);

        $table_name = 'users t';
        $join_tables = [
            ["usermeta t1" => 'LEFT'],
            ["usermeta t2" => 'LEFT'],
            ["usermeta t3" => 'LEFT'],
        ];
        $join_by = [
            "t.id = t1.user_id AND t1.meta_key = '{db_prefix}capabilities'",
            "t.id = t2.user_id AND t2.meta_key = 'g2fa_user_enabled'",
            "t.id = t3.user_id AND t3.meta_key = 'g2fa_user_secret'",
        ];
        $fields = [
            "t.id",
            "t.user_login",
            't.display_name',
            "REPLACE(REGEXP_SUBSTR(t1.meta_value, '\"(.*)\"'), '\"', '') AS role",
            "t2.meta_value AS g2fa_user_enabled",
            "t3.meta_value AS g2fa_user_secret"
        ];

        $roles = "'" . implode("','", self::WP_USER_ROLES) . "'";
        $having = $userRole !== 'another' ? "role = '{$userRole}'" : "role NOT IN ({$roles})";

        $model = new ClearModel();
        $users = $model->get_custom_data_from_tables($table_name, $join_tables, $join_by, [], '', $fields, $having);

        foreach ($users as $user) {
            $this->wpUsers[$user['id']] = $user;
        }
    }
    // Multi-user 2FA - end

    // Allowed IPs - start
    public function check_allowed_ips()
    {
        $binaryHelper = BinaryHelper::getInstance();
        $isEnabled = $this->isEnabledAllowedIps();

        if (!$isEnabled) {
            return;
        }

        if (!$binaryHelper->searchInFirewallList(MainHelper::getClientIp(), true)) {
            self::accessForbiddenError();
        }
    }

    public function isEnabledAllowedIps()
    {
        return ($this->binary_helper->searchInPermanentList(BinaryHelper::S) === false) ? SettingsModel::ALLOWED_IPS_DISABLED : $this->settingsM->get_by_name($this->settingsM->allowed_ips_enabled)->value_number ?? SettingsModel::ALLOWED_IPS_DISABLED;
    }

    public function changeAllowedIpsStatus(int $status) :array
    {
        if ($status) {
            $result = $this->settingsM->update_by_name($this->settingsM->allowed_ips_enabled, [
                $this->settingsM->settings_structure->value_number => SettingsModel::ALLOWED_IPS_ENABLED,
            ]);

            $errorMessage = __('Could not enable Allowed IPs', BW_PLUGIN_SLUG);

            if ($result) {
                $firewallWhitelistTableData = new FirewallWhitelistTableData();
                $firewallWhitelistTableData->prepare_items();
                if (!$firewallWhitelistTableData->items) {
                    $clientIp = MainHelper::getClientIp();
                    $binaryHelper = BinaryHelper::getInstance();
                    if (!$binaryHelper->addToFirewallWhitelist($clientIp)) {
                        $this->settingsM->update_by_name($this->settingsM->allowed_ips_enabled, [
                            $this->settingsM->settings_structure->value_number => SettingsModel::ALLOWED_IPS_DISABLED,
                        ]);
                        $result = false;
                        $errorMessage = __('Could not enable Allowed IPs. First, add your (or the desired) IP to the whitelist.', BW_PLUGIN_SLUG);
                    }
                }
            }
        } else {
            $result = $this->settingsM->update_by_name($this->settingsM->allowed_ips_enabled, [
                $this->settingsM->settings_structure->value_number => SettingsModel::ALLOWED_IPS_DISABLED,
            ]);
            $errorMessage = __('Could not disable Allowed IPs', BW_PLUGIN_SLUG);
        }

        if ($result !== false) {
            $response = ['result' => 'successful'];
            if ($status) {
                $response['html'] =
                    '<div class="bw-general-scan-settings-value-item bw-state-enabled green">' . __('Enabled', BW_PLUGIN_SLUG) . '</div>
                    <button class="js-bw-allowed-ips-disable bw-state-enabled button button-red">' . __('Disable', BW_PLUGIN_SLUG) . '</button>';
            } else {
                $response['html'] =
                    '<div class="bw-general-scan-settings-value-item bw-state-disabled red red1">' . __('Disabled', BW_PLUGIN_SLUG) . '</div>
                    <button class="js-bw-allowed-ips-enable bw-state-disabled button-primary">' . __('Enable', BW_PLUGIN_SLUG) . '</button>';
            }
        } else {
            $response = ['result' => 'error', 'message' => $errorMessage];
        }

        return $response;
    }

    public function checkAndDisableAllowedIps()
    {
        $allowedIpsEnabled = $this->settingsM->get_by_name($this->settingsM->allowed_ips_enabled)->value_number;

        if ($allowedIpsEnabled) {
            $firewallWhitelistTableData = new FirewallWhitelistTableData();
            $firewallWhitelistTableData->prepare_items();
            if (!$firewallWhitelistTableData->items) {
                $this->settingsM->update_by_name($this->settingsM->allowed_ips_enabled, [
                    $this->settingsM->settings_structure->value_number => SettingsModel::ALLOWED_IPS_DISABLED,
                ]);

                return true;
            }
        }

        return false;
    }
    // Allowed IPs - end

    public function getDefaultsProtectionSettings()
    {
        $settingsModel = new SettingsModel();
        return [
            $settingsModel->firewall_notfound_5m_limit => SettingsModel::FIREWALL_NOTFOUND_5M_LIMIT_DEFAULT,
            $settingsModel->firewall_notfound_24h_limit => SettingsModel::FIREWALL_NOTFOUND_24H_LIMIT_DEFAULT,

            $settingsModel->firewall_failed_login_5m_limit => SettingsModel::FIREWALL_FAILED_LOGIN_5M_LIMIT_DEFAULT,
            $settingsModel->firewall_failed_login_24h_limit => SettingsModel::FIREWALL_FAILED_LOGIN_24H_LIMIT_DEFAULT,

            $settingsModel->firewall_sql_injection_5m_limit => SettingsModel::FIREWALL_SQL_INJECTION_5M_LIMIT_DEFAULT,
            $settingsModel->firewall_sql_injection_24h_limit => SettingsModel::FIREWALL_SQL_INJECTION_24H_LIMIT_DEFAULT,

            $settingsModel->firewall_code_execution_5m_limit => SettingsModel::FIREWALL_CODE_EXECUTION_5M_LIMIT_DEFAULT,
            $settingsModel->firewall_code_execution_24h_limit => SettingsModel::FIREWALL_CODE_EXECUTION_24H_LIMIT_DEFAULT,

            $settingsModel->firewall_xss_5m_limit => SettingsModel::FIREWALL_XSS_5M_LIMIT_DEFAULT,
            $settingsModel->firewall_xss_24h_limit => SettingsModel::FIREWALL_XSS_24H_LIMIT_DEFAULT,

            $settingsModel->firewall_rfi_5m_limit => SettingsModel::FIREWALL_RFI_5M_LIMIT_DEFAULT,
            $settingsModel->firewall_rfi_24h_limit => SettingsModel::FIREWALL_RFI_24H_LIMIT_DEFAULT,

            $settingsModel->firewall_suspicious_file_uploads_5m_limit => SettingsModel::FIREWALL_SUSPICIOUS_FILE_UPLOADS_5M_LIMIT_DEFAULT,
            $settingsModel->firewall_suspicious_file_uploads_24h_limit => SettingsModel::FIREWALL_SUSPICIOUS_FILE_UPLOADS_24H_LIMIT_DEFAULT,
        ];
    }

}
