<?php
namespace betawall\helpers;

use betawall\db\models\FirewallBlacklistModel;
use betawall\db\models\FirewallWhitelistModel;
use betawall\db\models\MalwareModel;
use betawall\db\models\SettingsModel;
use Exception;
use PharData;

class BinaryHelper
{

    const IPV4_FILE = 'ipv4.bin';
    const IPV6_FILE = 'ipv6.bin';

    const WHITE_RANGE_IPV4_FILE = 'white-range-bot-ipv4.bin';
    const WHITE_RANGE_IPV6_FILE = 'white-range-bot-ipv6.bin';

    const BINARY_DB_DIRECTORY = BW_PLUGIN_PATH . 'db';
    const BINARY_TEMP_EXTRACT_DIRECTORY = 'update_temp';
    const BINARY_DB_UPDATE_ARCHIVE = 'db.tar.gz';

    const BINARY_FIREWALL_DIRECTORY = 'firewall';
    const BINARY_FIREWALL_BLACKLIST_DIRECTORY = 'blacklist';
    const BINARY_FIREWALL_WHITELIST_DIRECTORY = 'whitelist';

    const PROTOCOL_IPV4 = 0;
    const PROTOCOL_IPV6 = 1;

    const S1 = 1;
    const S0 = 0;
    const S = '2001:0db8:0000:0000:0000:0000:dead:beef';

    private static $instance;

    private function __construct() {}

    public static function getInstance(): BinaryHelper
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function extractTarGz(string $archivePath): bool
    {
        $wp_filesystem = FileSystemHelper::get_filesystem();

        $tempExtractToDir = self::BINARY_DB_DIRECTORY . '/' . self::BINARY_TEMP_EXTRACT_DIRECTORY;
        $tarPath = str_replace('.gz', '', $archivePath);

        if (!$wp_filesystem->is_dir($tempExtractToDir)) {
            if ($wp_filesystem->is_writable(self::BINARY_DB_DIRECTORY)) {
                @$wp_filesystem->mkdir($tempExtractToDir, FS_CHMOD_DIR, true);
            } else {
                return false;
            }
        }

        if (!$wp_filesystem->is_writable($tempExtractToDir)) {
            return false;
        }

        $tempIpV4 = $tempExtractToDir . '/' . self::IPV4_FILE;
        $tempIpV6 = $tempExtractToDir . '/' . self::IPV6_FILE;

        $tempWhiteRangeBotIpV4 = $tempExtractToDir . '/' . self::WHITE_RANGE_IPV4_FILE;
        $tempWhiteRangeBotIpV6 = $tempExtractToDir . '/' . self::WHITE_RANGE_IPV6_FILE;

        try {
            $isSuccess = true;

            $tar = new PharData($archivePath);
            $tar->decompress();

            $tarArchive = new PharData($tarPath);
            $tarArchive->extractTo($tempExtractToDir, null, true);

            $finalIpv4 = self::BINARY_DB_DIRECTORY . '/' . self::IPV4_FILE;
            $finalIpv6 = self::BINARY_DB_DIRECTORY . '/' . self::IPV6_FILE;

            $finalWhiteRangeBotIpv4 = self::BINARY_DB_DIRECTORY . '/' . self::WHITE_RANGE_IPV4_FILE;
            $finalWhiteRangeBotIpv6 = self::BINARY_DB_DIRECTORY . '/' . self::WHITE_RANGE_IPV6_FILE;

            if ($wp_filesystem->exists($tempIpV4) && $wp_filesystem->exists($tempIpV6)) {
                if ($wp_filesystem->exists($finalIpv4)) {
                    @$wp_filesystem->delete($finalIpv4);
                }
                if ($wp_filesystem->move($tempIpV4, $finalIpv4)) {
                    @$wp_filesystem->chmod($finalIpv4, 0777);
                }

                if ($wp_filesystem->exists($finalIpv6)) {
                    @$wp_filesystem->delete($finalIpv6);
                }
                if ($wp_filesystem->move($tempIpV6, $finalIpv6)) {
                    @$wp_filesystem->chmod($finalIpv6, 0777);
                }
            } else {
                $isSuccess = false;
            }

            if ($wp_filesystem->exists($tempWhiteRangeBotIpV4) && $wp_filesystem->exists($tempWhiteRangeBotIpV6)) {
                if ($wp_filesystem->exists($finalWhiteRangeBotIpv4)) {
                    @$wp_filesystem->delete($finalWhiteRangeBotIpv4);
                }
                if ($wp_filesystem->move($tempWhiteRangeBotIpV4, $finalWhiteRangeBotIpv4)) {
                    @$wp_filesystem->chmod($finalWhiteRangeBotIpv4, 0777);
                }

                if ($wp_filesystem->exists($finalWhiteRangeBotIpv6)) {
                    @$wp_filesystem->delete($finalWhiteRangeBotIpv6);
                }
                if ($wp_filesystem->move($tempWhiteRangeBotIpV6, $finalWhiteRangeBotIpv6)) {
                    @$wp_filesystem->chmod($finalWhiteRangeBotIpv6, 0777);
                }
            } else {
                $isSuccess = false;
            }

            $this->cleanTempFiles([
                $archivePath,
                $tarPath,
                $tempIpV4,
                $tempIpV6,
                $tempWhiteRangeBotIpV4,
                $tempWhiteRangeBotIpV6
            ]);

            return $isSuccess;
        } catch (Exception $e) {
            $this->cleanTempFiles([
                $archivePath,
                $tarPath,
                $tempIpV4,
                $tempIpV6,
                $tempWhiteRangeBotIpV4,
                $tempWhiteRangeBotIpV6
            ]);

            return false;
        }
    }

    private function cleanTempFiles(array $paths)
    {
        $wp_filesystem = FileSystemHelper::get_filesystem();

        foreach ($paths as $path) {
            if ($wp_filesystem->exists($path)) {
                $wp_filesystem->delete($path);
            }
        }
    }

    private function binaryCompare(string $data1, string $data2, int $length): int
    {
        for ($i = 0; $i < $length; $i++) {
            $diff = ord($data1[$i]) - ord($data2[$i]);
            if ($diff !== 0) {
                return $diff;
            }
        }
        return 0;
    }

    private function findInsertPosition($binaryFile, $ipBinary, $ipLength) {
        fseek($binaryFile, 0, SEEK_SET);

        while (($currentPos = ftell($binaryFile)) !== false) {
            $currentIp = fread($binaryFile, $ipLength);

            if ($currentIp === false || strlen($currentIp) < $ipLength) {
                break;
            }

            if ($currentIp > $ipBinary) {
                return $currentPos;
            }

            if ($currentIp === $ipBinary) {
                return false;
            }
        }

        return ftell($binaryFile);
    }

    private function searchEngine(string $filePath, string $targetHash, int $protocol = self::PROTOCOL_IPV4)
    {
        $wp_filesystem = FileSystemHelper::get_filesystem();

        if (!$wp_filesystem->exists($filePath)) {
            return false;
        }

        if (!$wp_filesystem->is_readable($filePath) || !$file = fopen($filePath, 'rb')) {
            $settingsModel = new SettingsModel();
            $settingsModel->update_by_name($settingsModel->access_state, [
                $settingsModel->settings_structure->value_number => SettingsModel::ACCESS_STATE_DENIED
            ]);
            return false;
        }

        $hashSize = ($protocol === self::PROTOCOL_IPV6) ? 16 : 4;

        fseek($file, 0, SEEK_END);
        $fileSize = ftell($file);

        if ($fileSize === 0) {
            fclose($file);
            return false;
        }

        $totalRecords = $fileSize / $hashSize;

        $left = 0;
        $right = $totalRecords - 1;

        while ($left <= $right) {
            $mid = floor(($left + $right) / 2);
            $midPos = $mid * $hashSize;
            fseek($file, $midPos);

            $hash = fread($file, $hashSize);

            if (strlen($hash) !== $hashSize) {
                fclose($file);
                return false;
            }

            $comparison = $this->binaryCompare($hash, $targetHash, $hashSize);

            if ($comparison === 0) {
                fclose($file);
                return $midPos;
            }

            if ($comparison < 0) {
                $left = $mid + 1;
            } else {
                $right = $mid - 1;
            }
        }

        fclose($file);
        return false;
    }

    public function searchInPermanentList(string $ip)
    {
        $wp_filesystem = FileSystemHelper::get_filesystem();

        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            $binaryFile = self::BINARY_DB_DIRECTORY . '/' .self::IPV4_FILE;
            if ($wp_filesystem->exists($binaryFile)) {
                $result = $this->searchEngine($binaryFile, inet_pton($ip));
                if ($result !== false) {
                    return ['db' => self::IPV4_FILE, 'offset' => $result];
                }
            }
            return false;
        }

        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            $binaryFile = self::BINARY_DB_DIRECTORY . '/' .self::IPV6_FILE;
            if ($wp_filesystem->exists($binaryFile)) {
                $result = $this->searchEngine($binaryFile, inet_pton($ip), self::PROTOCOL_IPV6);
                if ($result !== false) {
                    return ['db' => self::IPV6_FILE, 'offset' => $result];
                }
            }
            return false;
        }

        return false;
    }

    public function searchInFirewallList(string $ip, bool $inWhiteList = false)
    {
        $binaryFirewallDirectory = self::BINARY_DB_DIRECTORY . '/' . self::BINARY_FIREWALL_DIRECTORY;
        $binaryFirewallListDirectory = $binaryFirewallDirectory . '/' . self::BINARY_FIREWALL_BLACKLIST_DIRECTORY;
        if ($inWhiteList) {
            $binaryFirewallListDirectory = $binaryFirewallDirectory . '/' . self::BINARY_FIREWALL_WHITELIST_DIRECTORY;
        }

        $wp_filesystem = FileSystemHelper::get_filesystem();

        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            $binaryFile = $binaryFirewallListDirectory . '/' .self::IPV4_FILE;
            if ($wp_filesystem->exists($binaryFile)) {
                $result = $this->searchEngine($binaryFile, inet_pton($ip));
                if ($result !== false) {
                    return ['db' => self::IPV4_FILE, 'offset' => $result];
                }
            }
            return false;
        }

        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            $binaryFile = $binaryFirewallListDirectory . '/' .self::IPV6_FILE;
            if ($wp_filesystem->exists($binaryFile)) {
                $result = $this->searchEngine($binaryFile, inet_pton($ip), self::PROTOCOL_IPV6);
                if ($result !== false) {
                    return ['db' => self::IPV6_FILE, 'offset' => $result];
                }
            }
            return false;
        }

        return false;
    }

    public function addToFirewallBlacklist(string $ip, int $requestType): bool
    {
        if ($this->addToFirewallList($ip)) {
            $firewallBlacklistModel = new FirewallBlacklistModel();
            return $firewallBlacklistModel->add([
                $firewallBlacklistModel->firewall_blacklist_structure->type => $requestType,
                $firewallBlacklistModel->firewall_blacklist_structure->ip => $ip,
            ]);
        }
        return false;
    }

    public function addToFirewallWhitelist(string $ip): bool
    {
        $this->deleteFromFirewallList($ip);
        if ($this->addToFirewallList($ip, true)) {
            $firewallWhitelistModel = new FirewallWhitelistModel();
            return $firewallWhitelistModel->add([
                $firewallWhitelistModel->firewall_whitelist_structure->ip => $ip,
            ]);
        }
        return false;
    }

    public function moveFromFirewallBlacklistToWhitelist(string $ip): bool
    {
        if ($this->deleteFromFirewallList($ip) && $this->addToFirewallList($ip, true)) {
            $firewallWhitelistModel = new FirewallWhitelistModel();
            return $firewallWhitelistModel->add([
                $firewallWhitelistModel->firewall_whitelist_structure->ip => $ip,
            ]);
        }
        return false;
    }

    private function addToPermanentList(string $ip): bool
    {
        $wp_filesystem = FileSystemHelper::get_filesystem();

        if (!$wp_filesystem->is_dir(self::BINARY_DB_DIRECTORY)) {
            if ($wp_filesystem->is_writable(BW_PLUGIN_PATH)) {
                @$wp_filesystem->mkdir(self::BINARY_DB_DIRECTORY, FS_CHMOD_DIR, true);
            } else {
                return false;
            }
        }

        if ($wp_filesystem->is_writable(self::BINARY_DB_DIRECTORY)) {
            if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
                $binaryPermanentListFilePath = self::BINARY_DB_DIRECTORY . '/' . self::IPV4_FILE;
            } else if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
                $binaryPermanentListFilePath = self::BINARY_DB_DIRECTORY . '/' . self::IPV6_FILE;
            } else {
                return false;
            }

            if (!$binaryFile = @fopen($binaryPermanentListFilePath, 'c+b')) {
                $settingsModel = new SettingsModel();
                $settingsModel->update_by_name($settingsModel->access_state, [
                    $settingsModel->settings_structure->value_number => SettingsModel::ACCESS_STATE_DENIED
                ]);
                return false;
            }

            $ipBinary = inet_pton($ip);
            $ipLength = strlen($ipBinary);

            $insertPos = $this->findInsertPosition($binaryFile, $ipBinary, $ipLength);
            if ($insertPos === false) {
                fclose($binaryFile);
                return false;
            }

            fseek($binaryFile, 0, SEEK_END);
            $fileSize = ftell($binaryFile);

            for ($pos = $fileSize; $pos > $insertPos; $pos -= $ipLength) {
                fseek($binaryFile, $pos - $ipLength);
                $data = fread($binaryFile, $ipLength);
                fseek($binaryFile, $pos);
                fwrite($binaryFile, $data);
            }

            fseek($binaryFile, $insertPos);
            fwrite($binaryFile, $ipBinary);
            fclose($binaryFile);

            return true;
        }

        return false;
    }

    private function addToFirewallList(string $ip, $toWhitelist = false): bool
    {
        $wp_filesystem = FileSystemHelper::get_filesystem();

        if (!$wp_filesystem->is_dir(self::BINARY_DB_DIRECTORY)) {
            if ($wp_filesystem->is_writable(BW_PLUGIN_PATH)) {
                @$wp_filesystem->mkdir(self::BINARY_DB_DIRECTORY, FS_CHMOD_DIR, true);
            } else {
                return false;
            }
        }

        $binaryFirewallDirectory = self::BINARY_DB_DIRECTORY . '/' . self::BINARY_FIREWALL_DIRECTORY;
        if (!$wp_filesystem->is_dir($binaryFirewallDirectory)) {
            if ($wp_filesystem->is_writable(self::BINARY_DB_DIRECTORY)) {
                @$wp_filesystem->mkdir($binaryFirewallDirectory, FS_CHMOD_DIR, true);
            } else {
                return false;
            }
        }

        $binaryFirewallBlacklistDirectory = $binaryFirewallDirectory . '/' . self::BINARY_FIREWALL_BLACKLIST_DIRECTORY;
        if (!$wp_filesystem->is_dir($binaryFirewallBlacklistDirectory)) {
            if ($wp_filesystem->is_writable($binaryFirewallDirectory)) {
                @$wp_filesystem->mkdir($binaryFirewallBlacklistDirectory, FS_CHMOD_DIR, true);
            } else {
                return false;
            }
        }

        $binaryFirewallWhitelistDirectory = $binaryFirewallDirectory . '/' . self::BINARY_FIREWALL_WHITELIST_DIRECTORY;
        if (!$wp_filesystem->is_dir($binaryFirewallWhitelistDirectory)) {
            if ($wp_filesystem->is_writable($binaryFirewallDirectory)) {
                @$wp_filesystem->mkdir($binaryFirewallWhitelistDirectory, FS_CHMOD_DIR, true);
            } else {
                return false;
            }
        }

        if (!$toWhitelist && $this->searchInFirewallList($ip, true) !== false) {
            return false;
        }

        $finalDir = ($toWhitelist) ? $binaryFirewallWhitelistDirectory : $binaryFirewallBlacklistDirectory;
        if ($wp_filesystem->is_writable($finalDir)) {
            if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
                $binaryFirewallListFilePath = $finalDir . '/' . self::IPV4_FILE;
            } else if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
                $binaryFirewallListFilePath = $finalDir . '/' . self::IPV6_FILE;
            } else {
                return false;
            }

            if (!$binaryFile = @fopen($binaryFirewallListFilePath, 'c+b')) {
                $settingsModel = new SettingsModel();
                $settingsModel->update_by_name($settingsModel->access_state, [
                    $settingsModel->settings_structure->value_number => SettingsModel::ACCESS_STATE_DENIED
                ]);
                return false;
            }

            $ipBinary = inet_pton($ip);
            $ipLength = strlen($ipBinary);

            $insertPos = $this->findInsertPosition($binaryFile, $ipBinary, $ipLength);
            if ($insertPos === false) {
                fclose($binaryFile);
                return false;
            }

            fseek($binaryFile, 0, SEEK_END);
            $fileSize = ftell($binaryFile);

            for ($pos = $fileSize; $pos > $insertPos; $pos -= $ipLength) {
                fseek($binaryFile, $pos - $ipLength);
                $data = fread($binaryFile, $ipLength);
                fseek($binaryFile, $pos);
                fwrite($binaryFile, $data);
            }

            fseek($binaryFile, $insertPos);
            fwrite($binaryFile, $ipBinary);
            fclose($binaryFile);

            return true;
        }

        return false;
    }

    public function setS($state = self::S0)
    {
        $currentState = $this->searchInPermanentList(self::S);
        if ($currentState === false && $state == self::S1) {
            $this->addToPermanentList(self::S);
        } elseif ($currentState !== false && $state == self::S0) {
            $this->deleteFromPermatentList(self::S);
        }
    }

    private function deleteFromPermatentList($ip)
    {
        $binaryPermanentListDirectory = self::BINARY_DB_DIRECTORY;

        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            $protocol = self::PROTOCOL_IPV4;
            $binaryPermanentListFilePath = $binaryPermanentListDirectory . '/' . self::IPV4_FILE;
        } else if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            $protocol = self::PROTOCOL_IPV6;
            $binaryPermanentListFilePath = $binaryPermanentListDirectory . '/' . self::IPV6_FILE;
        } else {
            return false;
        }

        if (!$binaryFile = @fopen($binaryPermanentListFilePath, 'c+b')) {
            $settingsModel = new SettingsModel();
            $settingsModel->update_by_name($settingsModel->access_state, [
                $settingsModel->settings_structure->value_number => SettingsModel::ACCESS_STATE_DENIED
            ]);
            return false;
        }

        $ipBinary = inet_pton($ip);
        $ipLength = strlen($ipBinary);

        $deletePos = $this->searchEngine($binaryPermanentListFilePath, $ipBinary, $protocol);
        if ($deletePos !== false) {
            fseek($binaryFile, 0, SEEK_END);
            $fileSize = ftell($binaryFile);

            for ($pos = $deletePos + $ipLength; $pos < $fileSize; $pos += $ipLength) {
                fseek($binaryFile, $pos);
                $data = fread($binaryFile, $ipLength);
                fseek($binaryFile, $pos - $ipLength);
                fwrite($binaryFile, $data);
            }

            ftruncate($binaryFile, $fileSize - $ipLength);
        }
        fclose($binaryFile);

        return true;
    }

    public function deleteFromFirewallList(string $ip, bool $fromWhitelist = false): bool {
        $binaryFirewallListDirectory = self::BINARY_DB_DIRECTORY . '/' . self::BINARY_FIREWALL_DIRECTORY;

        if ($fromWhitelist) {
            $binaryFirewallListDirectory .= '/' . self::BINARY_FIREWALL_WHITELIST_DIRECTORY;
        } else {
            $binaryFirewallListDirectory .= '/' . self::BINARY_FIREWALL_BLACKLIST_DIRECTORY;
        }

        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            $protocol = self::PROTOCOL_IPV4;
            $binaryFirewallFilePath = $binaryFirewallListDirectory . '/' . self::IPV4_FILE;
        } else if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            $protocol = self::PROTOCOL_IPV6;
            $binaryFirewallFilePath = $binaryFirewallListDirectory . '/' . self::IPV6_FILE;
        } else {
            return false;
        }

        if (!$binaryFile = @fopen($binaryFirewallFilePath, 'c+b')) {
            $settingsModel = new SettingsModel();
            $settingsModel->update_by_name($settingsModel->access_state, [
                $settingsModel->settings_structure->value_number => SettingsModel::ACCESS_STATE_DENIED
            ]);
            return false;
        }

        $ipBinary = inet_pton($ip);
        $ipLength = strlen($ipBinary);

        $deletePos = $this->searchEngine($binaryFirewallFilePath, $ipBinary, $protocol);
        if ($deletePos !== false) {
            fseek($binaryFile, 0, SEEK_END);
            $fileSize = ftell($binaryFile);

            for ($pos = $deletePos + $ipLength; $pos < $fileSize; $pos += $ipLength) {
                fseek($binaryFile, $pos);
                $data = fread($binaryFile, $ipLength);
                fseek($binaryFile, $pos - $ipLength);
                fwrite($binaryFile, $data);
            }

            ftruncate($binaryFile, $fileSize - $ipLength);
        }
        fclose($binaryFile);

        if ($fromWhitelist) {
            $firewallWhitelistModel = new FirewallWhitelistModel();
            if (!$firewallWhitelistModel->delete_by_ip($ip)) {
                return false;
            }
        } else {
            $firewallBlacklistModel = new FirewallBlacklistModel();
            if (!$firewallBlacklistModel->delete_by_ip($ip)) {
                return false;
            }
            $malwareModel = new MalwareModel();
            $malwareModel->delete_by_ip($ip);
        }

        return true;
    }

    public function checkPermissionsAccess(): bool
    {
        $directories = [
            self::BINARY_DB_DIRECTORY,
            self::BINARY_DB_DIRECTORY . '/' . self::BINARY_TEMP_EXTRACT_DIRECTORY,
            self::BINARY_DB_DIRECTORY . '/' .self::BINARY_FIREWALL_DIRECTORY,
            self::BINARY_DB_DIRECTORY . '/' .self::BINARY_FIREWALL_DIRECTORY . '/' .self::BINARY_FIREWALL_BLACKLIST_DIRECTORY,
            self::BINARY_DB_DIRECTORY . '/' .self::BINARY_FIREWALL_DIRECTORY . '/' .self::BINARY_FIREWALL_WHITELIST_DIRECTORY
        ];

        $files = [
            self::BINARY_DB_DIRECTORY . '/' . self::IPV4_FILE,
            self::BINARY_DB_DIRECTORY . '/' . self::IPV6_FILE,

            self::BINARY_DB_DIRECTORY . '/' . self::WHITE_RANGE_IPV4_FILE,
            self::BINARY_DB_DIRECTORY . '/' . self::WHITE_RANGE_IPV6_FILE,

            self::BINARY_DB_DIRECTORY . '/' .self::BINARY_FIREWALL_DIRECTORY. '/' . self::BINARY_FIREWALL_BLACKLIST_DIRECTORY . '/' . self::IPV4_FILE,
            self::BINARY_DB_DIRECTORY . '/' .self::BINARY_FIREWALL_DIRECTORY. '/' . self::BINARY_FIREWALL_BLACKLIST_DIRECTORY . '/' . self::IPV6_FILE,

            self::BINARY_DB_DIRECTORY . '/' .self::BINARY_FIREWALL_DIRECTORY. '/' . self::BINARY_FIREWALL_WHITELIST_DIRECTORY . '/' . self::IPV4_FILE,
            self::BINARY_DB_DIRECTORY . '/' .self::BINARY_FIREWALL_DIRECTORY. '/' . self::BINARY_FIREWALL_WHITELIST_DIRECTORY . '/' . self::IPV6_FILE,
        ];

        $wp_filesystem = FileSystemHelper::get_filesystem();

        foreach ($directories as $directory) {
            if (!$wp_filesystem->is_dir($directory)) {
                return false;
            }

            if (!$wp_filesystem->is_readable($directory)) {
                return false;
            }

            $tempFile = "$directory/test.tmp";
            if (!$wp_filesystem->put_contents($tempFile, 'test')) {
                return false;
            }

            $wp_filesystem->delete($tempFile);
        }

        foreach ($files as $file) {
            if ($wp_filesystem->exists($file)) {
                if (!$wp_filesystem->is_readable($file)) {
                    return false;
                }

                if (!$wp_filesystem->is_writable($file)) {
                    return false;
                }
            }
        }

        return true;
    }

    public function initSysFiles()
    {
        $wp_filesystem = FileSystemHelper::get_filesystem();

        $data = [
            'directory' => self::BINARY_DB_DIRECTORY,
            'temp_directory' => self::BINARY_DB_DIRECTORY . '/' . self::BINARY_TEMP_EXTRACT_DIRECTORY,
            'firewall' => [
                'directory' => self::BINARY_DB_DIRECTORY . '/' . self::BINARY_FIREWALL_DIRECTORY,
                'blacklist' => [
                    'directory' => self::BINARY_DB_DIRECTORY . '/' . self::BINARY_FIREWALL_DIRECTORY . '/' . self::BINARY_FIREWALL_BLACKLIST_DIRECTORY,
                    'ipv4_file' => self::BINARY_DB_DIRECTORY . '/' . self::BINARY_FIREWALL_DIRECTORY . '/' . self::BINARY_FIREWALL_BLACKLIST_DIRECTORY . '/' . self::IPV4_FILE,
                    'ipv6_file' => self::BINARY_DB_DIRECTORY . '/' . self::BINARY_FIREWALL_DIRECTORY . '/' . self::BINARY_FIREWALL_BLACKLIST_DIRECTORY . '/' . self::IPV6_FILE,
                    'model' => new FirewallBlacklistModel(),
                ],
                'whitelist' => [
                    'directory' => self::BINARY_DB_DIRECTORY . '/' . self::BINARY_FIREWALL_DIRECTORY . '/' . self::BINARY_FIREWALL_WHITELIST_DIRECTORY,
                    'ipv4_file' => self::BINARY_DB_DIRECTORY . '/' . self::BINARY_FIREWALL_DIRECTORY . '/' . self::BINARY_FIREWALL_WHITELIST_DIRECTORY . '/' . self::IPV4_FILE,
                    'ipv6_file' => self::BINARY_DB_DIRECTORY . '/' . self::BINARY_FIREWALL_DIRECTORY . '/' . self::BINARY_FIREWALL_WHITELIST_DIRECTORY . '/' . self::IPV6_FILE,
                    'model' => new FirewallWhitelistModel(),
                ],
            ],
            'ipv4_file' => self::BINARY_DB_DIRECTORY . '/' . self::IPV4_FILE,
            'ipv6_file' => self::BINARY_DB_DIRECTORY . '/' . self::IPV6_FILE,
            'white_range_bot_ipv4_file' => self::BINARY_DB_DIRECTORY . '/' . self::WHITE_RANGE_IPV4_FILE,
            'white_range_bot_ipv6_file' => self::BINARY_DB_DIRECTORY . '/' . self::WHITE_RANGE_IPV6_FILE,
        ];

        $this->createDirectoriesAndFiles($wp_filesystem, $data);
        $this->syncFirewallFiles($data['firewall']);
    }

    private function createDirectoriesAndFiles($wp_filesystem, $data)
    {
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                $this->createDirectoriesAndFiles($wp_filesystem, $value);
            } elseif (preg_match('/directory$/', $key)) {
                if (!$wp_filesystem->is_dir($value)) {
                    @$wp_filesystem->mkdir($value, 0777);
                }
            } elseif (preg_match('/_file$/', $key)) {
                @$wp_filesystem->put_contents($value, '');
                @$wp_filesystem->chmod($value, 0777);
            }
        }
    }

    private function syncFirewallFiles($firewallData)
    {
        foreach (['blacklist', 'whitelist'] as $type) {
            if (!isset($firewallData[$type]['model'])) {
                continue;
            }

            $model = $firewallData[$type]['model'];

            $ipsList = $model->get_all();
            if (!empty($ipsList)) {
                foreach ($ipsList as $item) {
                    $this->addToFirewallList($item->ip, $type === 'whitelist');
                }
            }
        }
    }

    public function getCountIpsInFile($protocol, $inWhitelist = false)
    {
        if ($protocol == self::PROTOCOL_IPV4) {
            $recordSize = 4;
            if (!$inWhitelist) {
                $file = self::BINARY_DB_DIRECTORY . '/' . self::BINARY_FIREWALL_DIRECTORY . '/' . self::BINARY_FIREWALL_BLACKLIST_DIRECTORY . '/' . self::IPV4_FILE;
            } else {
                $file = self::BINARY_DB_DIRECTORY . '/' . self::BINARY_FIREWALL_DIRECTORY . '/' . self::BINARY_FIREWALL_WHITELIST_DIRECTORY . '/' . self::IPV4_FILE;
            }
        } else if ($protocol == self::PROTOCOL_IPV6) {
            $recordSize = 16;
            if (!$inWhitelist) {
                $file = self::BINARY_DB_DIRECTORY . '/' . self::BINARY_FIREWALL_DIRECTORY . '/' . self::BINARY_FIREWALL_BLACKLIST_DIRECTORY . '/' . self::IPV6_FILE;
            } else {
                $file = self::BINARY_DB_DIRECTORY . '/' . self::BINARY_FIREWALL_DIRECTORY . '/' . self::BINARY_FIREWALL_WHITELIST_DIRECTORY . '/' . self::IPV6_FILE;
            }
        } else {
            return false;
        }

        $wp_filesystem = FileSystemHelper::get_filesystem();
        if ($wp_filesystem->exists($file)) {
            $fileSize = filesize($file);
            return ($fileSize > 0) ? intdiv($fileSize, $recordSize) : 0;
        }

        return 0;
    }

    public function isIpInRange(string $ip): bool
    {
        $binIp = inet_pton($ip);
        if (!$binIp) {
            return false;
        }

        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            $file = self::BINARY_DB_DIRECTORY . '/' . self::WHITE_RANGE_IPV6_FILE;
            $ipLength = 16;
        } elseif (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            $file = self::BINARY_DB_DIRECTORY . '/' . self::WHITE_RANGE_IPV4_FILE;
            $ipLength = 4;
        } else {
            return false;
        }

        $recordSize = $ipLength * 2;
        if (!file_exists($file)) {
            return false;
        }

        $handle = fopen($file, 'rb');
        fseek($handle, 0, SEEK_END);
        $total = ftell($handle) / $recordSize;
        $low = 0;
        $high = $total - 1;

        while ($low <= $high) {
            $mid = (int)(($low + $high) / 2);
            fseek($handle, $mid * $recordSize);
            $start = fread($handle, $ipLength);
            $end = fread($handle, $ipLength);

            if ($binIp < $start) {
                $high = $mid - 1;
            } elseif ($binIp > $end) {
                $low = $mid + 1;
            } else {
                fclose($handle);
                return true;
            }
        }

        fclose($handle);
        return false;
    }

}