<?php
namespace betawall\db;

class DatabaseProvider
{

    protected $wpdb;
    protected $table_name;

    protected function __construct($table_name)
    {
        global $wpdb;
        $this->wpdb = $wpdb;
        $this->table_name = $table_name ? ($wpdb->prefix . $table_name) : '';
    }

    protected function create_table($sql)
    {
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    protected function delete_table()
    {
        return $this->wpdb->query("DROP TABLE IF EXISTS $this->table_name");
    }

    protected function get_data($where, $operator, $output_type, $limit = null, $offset = null, $order_rule = null)
    {
        $query = "SELECT * FROM {$this->table_name}";
        $query_params = [];

        if (!empty($where)) {
            $conditions = [];
            foreach ($where as $column => $value) {
                if ($column === 'time_range') {
                    list($startTime, $endTime) = $value;
                    $conditions[] = "created_at BETWEEN %d AND %d";
                    $query_params[] = $startTime;
                    $query_params[] = $endTime;
                } else if ($column === 'like') {
                    list($field, $search) = $value;
                    $conditions[] = "{$field} LIKE %s";
                    $query_params[] = '%' . $search . '%';
                } elseif (is_array($value)) {
                    $placeholders = implode(', ', array_fill(0, count($value), '%s'));
                    $conditions[] = "{$column} IN ($placeholders)";
                    foreach ($value as $val) {
                        $query_params[] = $val;
                    }
                } else {
                    $conditions[] = "{$column} = %s";
                    $query_params[] = $value;
                }
            }
            $query .= " WHERE " . implode(" $operator ", $conditions);
        }

        if (!empty($order_rule)) {
            $query .= " ORDER BY " . esc_sql($order_rule['column']) . " " . (strtoupper($order_rule['direction']) === 'DESC' ? 'DESC' : 'ASC');
        }

        if ($limit !== null) {
            $query .= " LIMIT %d";
            $query_params[] = $limit;
            if ($offset !== null) {
                $query .= " OFFSET %d";
                $query_params[] = $offset;
            }
        }

        if (empty($query_params)) {
            return $this->wpdb->get_results($query, $output_type);
        }

        $prepared_query = $this->wpdb->prepare($query, $query_params);
        return $this->wpdb->get_results($prepared_query, $output_type);
    }

    protected function get_data_with_other_table_condition($table_field, $other_table_name, $other_table_field, $where, $operator, $output_type, $limit)
    {
        $query = "SELECT * FROM $this->table_name AS t1  WHERE t1.$table_field IN (SELECT $other_table_field FROM {$this->wpdb->prefix}$other_table_name)";

        if (!empty($where)) {
            $conditions = [];
            foreach ($where as $column => $value) {
                $conditions[] = $this->wpdb->prepare("t1.$column = %s", $value);
            }
            $query .= " AND " . implode(" $operator ", $conditions);
        }

        if ($limit !== null) {
            $query .= " LIMIT %d";
            $query = $this->wpdb->prepare($query, $limit);
        }

        return $this->wpdb->get_results($query, $output_type);
    }

    protected function get_counts($where, $operator)
    {
        $query = "SELECT COUNT(id) FROM $this->table_name";
        $conditions = [];

        foreach ($where as $column => $value) {
            if ($column === 'time_range') {
                if (is_array($value) && count($value) === 2) {
                    list($startTime, $endTime) = $value;
                    $conditions[] = $this->wpdb->prepare("created_at BETWEEN %d AND %d", $startTime, $endTime);
                }
            } elseif ($column === 'like') {
                list($searchColumn, $searchValue) = $value;
                $conditions[] = $this->wpdb->prepare("$searchColumn LIKE %s", '%' . $this->wpdb->esc_like($searchValue) . '%');
            } else {
                $conditions[] = $this->wpdb->prepare("$column = %s", $value);
            }
        }

        if (!empty($conditions)) {
            $query .= " WHERE " . implode(" $operator ", $conditions);
        }
        return $this->wpdb->get_var($query);
    }

    protected function insert_data($data)
    {
        $data['created_at'] = current_time('timestamp');
        $data['updated_at'] = current_time('timestamp');
        return $this->wpdb->insert($this->table_name, $data);
    }

    protected function update_data($data, $where)
    {
        $data['updated_at'] = current_time('timestamp');
        return $this->wpdb->update($this->table_name, $data, $where);
    }

    protected function update_multiple_with_same_value($column, $value, $where_column, $where_values)
    {
        $placeholders = implode(',', array_fill(0, count($where_values), '%s'));

        $sql = sprintf(
            "UPDATE `%s` SET `%s` = %%s WHERE `%s` IN (%s)",
            $this->table_name,
            $column,
            $where_column,
            $placeholders
        );

        $prepared_query = $this->wpdb->prepare($sql, array_merge([$value], $where_values));
        $result = $this->wpdb->query($prepared_query);

        if ($result !== false && $this->wpdb->rows_affected === 0) {
            return 0;
        }

        return $result;
    }

    protected function delete_data_limit($where, $limit = null)
    {
        $conditions = [];
        $values = [];

        foreach ($where as $key => $value) {
            $conditions[] = "`$key` = %s";
            $values[] = $value;
        }

        $sql = "DELETE FROM `{$this->table_name}` WHERE " . implode(' AND ', $conditions);

        if ($limit !== null) {
            $sql .= " LIMIT %d";
            $values[] = $limit;
        }

        $prepared = $this->wpdb->prepare($sql, ...$values);
        return $this->wpdb->query($prepared);
    }

    protected function delete_data($where)
    {
        return $this->wpdb->delete($this->table_name, $where);
    }

    protected function delete_data_multiple_ids(array $ids)
    {
        if (empty($ids)) {
            return false;
        }

        $placeholders = implode(',', array_fill(0, count($ids), '%d'));
        $query = "DELETE FROM $this->table_name WHERE id IN ($placeholders)";

        return $this->wpdb->query($this->wpdb->prepare($query, ...$ids));
    }

    protected function update_data_multiple_ids($column, $value, $ids)
    {
        if (empty($ids)) {
            return false;
        }

        $placeholders = implode(',', array_fill(0, count($ids), '%d'));

        $sql = sprintf(
            "UPDATE `%s` SET `%s` = %%s WHERE `id` IN (%s)",
            $this->table_name,
            $column,
            $placeholders
        );

        $prepared_query = $this->wpdb->prepare($sql, array_merge([$value], $ids));
        $result = $this->wpdb->query($prepared_query);

        if ($result !== false && $this->wpdb->rows_affected === 0) {
            return 0;
        }

        return $result;
    }

    protected function get_custom_data_from_tables(string $table_name, array $join_tables = [], array $join_by = [], array $where = [], string $operator = 'AND', array $fields = [], string $having = '') :array
    {
        $prefix = $this->wpdb->prefix;
        $conditions = [];

        $sqlFields = $fields ? implode(', ', $fields) : '*';

        $sql = "SELECT {$sqlFields} FROM {$prefix}{$table_name}";

        foreach ($join_tables as $k => $arr) {

            $table = key($arr);
            $connection = $arr[$table] ?: '';

            if (strpos($join_by[$k], '{db_prefix}') !== false) {
                $join_by[$k] = str_replace('{db_prefix}', $prefix, $join_by[$k]);
            } else {
                $join_by[$k] = str_replace('{db_prefix}', 'wp_', $join_by[$k]);
            }

            $sql.= " {$connection} JOIN {$prefix}{$table} ON {$join_by[$k]}";
        }

        if (!empty($where)) {
            $sql.= " WHERE ";
            foreach ($where as $field => $value) {
                if (strpos($value, '{db_prefix}') === 0) {
                    $value = str_replace('{db_prefix}', $prefix, $value);
                } else {
                    $value = str_replace('{db_prefix}', 'wp_', $value);
                }
                $conditions[] = $this->wpdb->prepare("{$field} = %s", $value);
            }
            $sql.=  implode(" {$operator} ", $conditions);
        }

        if ($having) {
            $sql.=  "\nHAVING {$having}";
        }

        return $this->wpdb->get_results($sql, ARRAY_A);
    }

    protected function get_custom_data_from_tables_clear(string $sql) :array
    {
        $prefix = $this->wpdb->prefix;
        $sql = str_replace('{db_prefix}', $prefix, $sql);

        return $this->wpdb->get_results($sql, ARRAY_A);
    }

    protected function delete_custom_data_multiple_ids(string $table, string $field, array $ids, array $where) :int
    {
        if (empty($ids)) {
            return 0;
        }
        $prefix = $this->wpdb->prefix;

        $placeholders = implode(',', array_fill(0, count($ids), '%d'));
        $sql = "DELETE FROM {$prefix}{$table} WHERE {$field} IN ({$placeholders})";

        if (!empty($where)) {
            $conditions = [];
            foreach ($where as $whereItem) {
                $field = key($whereItem);
                $value = $whereItem[$field];

                $conditions[] = $this->wpdb->prepare("{$field} = %s", $value);
            }
            $sql.= ' AND (' . implode(" OR ", $conditions) . ');';
        }

//        result query string example
//        DELETE FROM table t
//        WHERE t.field IN (%d,%d) AND (t.field1 = 'value1' OR t.field1 = 'value2');

        return $this->wpdb->query($this->wpdb->prepare($sql, ...$ids));
    }

    protected function insert_into_custom_table(string $table, array $fields) :int
    {
        return (int) $this->wpdb->insert($this->wpdb->prefix.$table, $fields);
    }
}
