<?php
namespace betawall\crud;

if (!class_exists('WP_List_Table')) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

use betawall\db\models\FirewallBlacklistModel;
use betawall\db\models\MalwareModel;
use betawall\helpers\FirewallHelper;
use WP_List_Table;

class FirewallBlacklistTableData extends WP_List_Table
{

    use CustomPaginationTrait;
    protected $pagination_param = 'blacklist_paged';

    const FILTER_DATE_RANGE_5M = '5m';
    const FILTER_DATE_RANGE_24H = '24h';
    const FILTER_DATE_RANGE_7D = '7d';
    const FILTER_DATE_RANGE_30D = '30d';

    const PER_PAGE = 20;

    private function filter_request_type_validation()
    {
        return [
            MalwareModel::TYPE_NOT_FOUND,
            MalwareModel::TYPE_LOGIN,
            MalwareModel::TYPE_SQL_INJECTION,
            MalwareModel::TYPE_CODE_EXECUTION,
            MalwareModel::TYPE_XSS,
            MalwareModel::TYPE_RFI,
            MalwareModel::TYPE_SUSPICIOUS_FILE
        ];
    }

    private function filter_date_range_validation()
    {
        return [
            self::FILTER_DATE_RANGE_5M,
            self::FILTER_DATE_RANGE_24H,
            self::FILTER_DATE_RANGE_7D,
            self::FILTER_DATE_RANGE_30D
        ];
    }

    public function __construct()
    {
        parent::__construct([
            'singular' => 'item',
            'plural' => 'items',
            'ajax' => false,
        ]);
    }

    public function get_columns()
    {
        return [
            'ip' => __('IP Address', BW_PLUGIN_SLUG),
            'protocol' => __('Protocol', BW_PLUGIN_SLUG),
            'request_type' => __('Threat Type', BW_PLUGIN_SLUG),
            'created_at' => __('Added', BW_PLUGIN_SLUG),
            'actions' => __('Actions', BW_PLUGIN_SLUG),
        ];
    }

    public function get_sortable_columns()
    {
        return [
            'created_at' => ['blacklist_created_at', false],
        ];
    }

    public function column_default($item, $column_name)
    {
        return $item[$column_name];
    }

    protected function column_ip($item)
    {
        $ip = esc_html($item['ip']);
        $search_query = isset($_GET['blacklist_search']) ? esc_html($_GET['blacklist_search']) : '';

        if (!empty($search_query) && strpos($ip, $search_query) !== false) {
            $highlighted = sprintf('<span class="bw-query-contains">%s</span>', esc_html($search_query));
            $ip = str_replace($search_query, $highlighted, $ip);
        }

        return sprintf('<span class="bw-table-data-ip">%s</span>', $ip);
    }

    protected function column_protocol($item)
    {
        if (filter_var($item['ip'], FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            return 'IPv4';
        }
        if (filter_var($item['ip'], FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            return 'IPv6';
        }
        return 'Invalid IP';
    }

    protected function column_request_type($item)
    {
        switch ($item['request_type']) {
            case MalwareModel::TYPE_NOT_FOUND: return __('404 Request Flood', BW_PLUGIN_SLUG);
            case MalwareModel::TYPE_LOGIN: return __('Admin Panel Brute Force', BW_PLUGIN_SLUG);
            case MalwareModel::TYPE_SQL_INJECTION: return __('SQL Injection Attacks', BW_PLUGIN_SLUG);
            case MalwareModel::TYPE_CODE_EXECUTION: return __('Arbitrary Code Execution', BW_PLUGIN_SLUG);
            case MalwareModel::TYPE_XSS: return __('XSS Attacks', BW_PLUGIN_SLUG);
            case MalwareModel::TYPE_RFI: return __('RFI Attacks', BW_PLUGIN_SLUG);
            case MalwareModel::TYPE_SUSPICIOUS_FILE: return __('Suspicious File Uploads', BW_PLUGIN_SLUG);
            default: return '';
        }
    }

    protected function column_created_at($item)
    {
        return date('Y-m-d H:i:s', $item['created_at']);
    }

    protected function column_actions($item)
    {
        $buttons = '
            <a href="#" class="button-primary bw-firewall-blacklist-move-item-to-whitelist" data-id="%s">%s</a>
            <a href="#" class="button bw-firewall-blacklist-delete-item" data-id="%s">%s</a> 
        ';
        return sprintf(
            $buttons,
            $item['id'],
            esc_html(__('Move to Whitelist', BW_PLUGIN_SLUG)),
            $item['id'],
            esc_html(__('Delete', BW_PLUGIN_SLUG))
        );
    }

    public function display_tablenav($which)
    {
        if ($which === 'bottom') {
            parent::display_tablenav($which);
        }
    }

    public function prepare_items()
    {
        $order_by = isset($_GET['orderby']) ? sanitize_text_field($_GET['orderby']) : null;
        $order = isset($_GET['order']) && strtolower($_GET['order']) === 'desc' ? 'DESC' : 'ASC';

        $filter_request_type = isset($_GET['filter_request_type']) ? (int)sanitize_text_field($_GET['filter_request_type']) : null;
        $filter_date_range = isset($_GET['filter_date_range']) ? sanitize_text_field($_GET['filter_date_range']) : null;

        $blacklist_search = isset($_GET['blacklist_search']) ? sanitize_text_field($_GET['blacklist_search']) : null;

        $this->_column_headers = [$this->get_columns(), [], $this->get_sortable_columns()];

        $order_rule = ['column' => 'created_at', 'direction' => 'DESC'];
        if (!empty($order_by) && $order_by === 'blacklist_created_at') {
            $order_rule['direction'] = $order;
        }

        $filter_request_type_rule = null;
        if (in_array($filter_request_type, $this->filter_request_type_validation(), true)) {
            $filter_request_type_rule = $filter_request_type;
        }

        $filter_date_range_rule = null;
        if (in_array($filter_date_range, $this->filter_date_range_validation(), true)) {
            switch ($filter_date_range) {
                case self::FILTER_DATE_RANGE_5M:
                    $filter_date_range_rule = [
                        current_time('timestamp') - FirewallHelper::FIVE_MINUTES,
                        current_time('timestamp')
                    ];
                    break;
                case self::FILTER_DATE_RANGE_24H:
                    $filter_date_range_rule = [
                        current_time('timestamp') - FirewallHelper::TWENTY_FOUR_HOURS,
                        current_time('timestamp')
                    ];
                break;
                case self::FILTER_DATE_RANGE_7D:
                    $filter_date_range_rule = [
                        current_time('timestamp') - (FirewallHelper::TWENTY_FOUR_HOURS * 7),
                        current_time('timestamp')
                    ];
                break;
                case self::FILTER_DATE_RANGE_30D:
                    $filter_date_range_rule = [
                        current_time('timestamp') - (FirewallHelper::TWENTY_FOUR_HOURS * 30),
                        current_time('timestamp')
                    ];
                break;
            }
        }

        $firewallBlacklistModel = new FirewallBlacklistModel();
        $totalCountBlacklist = $firewallBlacklistModel->get_crud_table_count(
            $filter_request_type_rule, $filter_date_range_rule, $blacklist_search
        );

        $currentPage = $this->get_pagenum();
        $totalPages = (int) ceil($totalCountBlacklist / self::PER_PAGE);
        if ($currentPage > $totalPages) {
            $currentPage = $totalPages;
        }

        $offset = ($currentPage - 1) * self::PER_PAGE;
        if ($offset < 0) {
            $offset = 0;
        }

        $dataBlacklist = $firewallBlacklistModel->get_crud_table_data(
            ARRAY_A, self::PER_PAGE, $offset, $order_rule, $filter_request_type_rule, $filter_date_range_rule, $blacklist_search
        );
        $this->items = !empty($dataBlacklist) ? $dataBlacklist : [];

        $this->set_pagination_args([
            'total_items' => $totalCountBlacklist,
            'per_page' => self::PER_PAGE,
        ]);
    }

}
