<?php
namespace betawall\controllers;

use betawall\components\ViewManager;
use betawall\db\models\SettingsModel;
use betawall\helpers\FileSystemHelper;
use betawall\helpers\MainHelper;
use betawall\helpers\SecurityHelper;

class GeneralScanController
{

    const VIEWS_DIR_NAME = 'general_scan';

    /** @var MainHelper */
    public $mainHelper;

    public function __construct()
    {
        $this->mainHelper = new MainHelper();
    }

    public function index()
    {
        $settingsModel = new SettingsModel();
        $settingsModel->update_by_name($settingsModel->general_scan_red_dot, [
            $settingsModel->settings_structure->value_number => current_time('timestamp')
        ]);

        $viewManager = new ViewManager();
        $viewManager->render(self::VIEWS_DIR_NAME, '_index', []);
    }

    /**
     * AJAX action
     */
    public function scan_data()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $hiddenUsersRaw = SecurityHelper::detect_hidden_users();
        $hiddenUserCreators = SecurityHelper::find_user_creators($hiddenUsersRaw);
        $hooksForHiddenUsers = SecurityHelper::scan_hooks_for_hidden_users($hiddenUsersRaw);
        $hiddenUsers = [];
        foreach ($hiddenUsersRaw as $k => $hiddenUser) {
            unset($hiddenUser['meta_keys']);
            $hiddenUsers[$hiddenUser['ID']] = $hiddenUser;
        }

        foreach ($hiddenUserCreators as $hiddenUserCreator) {
            if (!empty($hiddenUserCreator['source_user'])) {
                $userId = $hiddenUserCreator['source_user']['ID'];
                if (isset($hiddenUsers[$userId])) {
                    unset($hiddenUserCreator['source_user']);
                    $hiddenUsers[$userId]['creator'] = $hiddenUserCreator;
                }
            }
        }

        foreach ($hooksForHiddenUsers as $hooksForHiddenUser) {
            if (!empty($hooksForHiddenUser['source_user'])) {
                $userId = $hooksForHiddenUser['source_user']['ID'];
                if (isset($hiddenUsers[$userId])) {
                    unset($hooksForHiddenUser['source_user']);
                    $hiddenUsers[$userId]['hook'] = $hooksForHiddenUser;
                }
            }
        }

        $viewManager = new ViewManager();
        $html = $viewManager->renderPartial(self::VIEWS_DIR_NAME, '_scan_data', [
            'hidden_users'            => $hiddenUsers,
            'hidden_plugins'          => SecurityHelper::scan_plugin_hiding(),
            'ssl_status'              => $this->mainHelper->checkSiteSSL(home_url()),
            'readme_and_license'      => $this->mainHelper->detectReadmeAndLicense(),
            'debug_log'               => $this->mainHelper->detectDebugLog(),
            'rest_api_status'         => $this->mainHelper->checkIsEnabledRestApi(),
            'rest_api_external_usage' => $this->mainHelper->checkForExternalUsage(MainHelper::TYPE_REST_API),
            'xml_rpc_status'          => $this->mainHelper->checkIsEnabledXmlRpc(),
            'xml_rpc_external_usage'  => $this->mainHelper->checkForExternalUsage(MainHelper::TYPE_XML_RPC),
            'rss_status'              => $this->mainHelper->checkIsEnabledRss(),
            'rss_external_usage'      => $this->mainHelper->checkForExternalUsage(MainHelper::TYPE_RSS),
            'unused_themes'           => $this->mainHelper->getUnusedThemes(),
            'unused_plugins'          => $this->mainHelper->getUnusedPlugins(),
        ], true);

        $response = [
            'result' => 'successful',
            'html'   => $html,
        ];

        wp_send_json($response);
    }

    /**
     * AJAX action
     */
    public function delete_hidden_user()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $userId = isset($_POST['user_id']) ? sanitize_text_field($_POST['user_id']) : null;

        if ($userId) {
            if (get_userdata($userId)) {
                wp_delete_user($userId);
                $response = ['result' => 'successful'];
            } else {
                $response = ['result' => 'error', 'message' => __("User ID {$userId} not found", BW_PLUGIN_SLUG)];
            }
        } else {
            $response = ['result' => 'error', 'message' => __('Incorrect user_id', BW_PLUGIN_SLUG)];
        }

        wp_send_json($response);
    }

    /**
     * AJAX action
     */
    public function delete_file()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $type = isset($_POST['type']) ? sanitize_text_field($_POST['type']) : '';

        $wp_filesystem = FileSystemHelper::get_filesystem();
        $isSuccessOperation = false;
        $message = '';

        switch ($type) {
            case 'readme':
                $readmeFile = ABSPATH . 'readme.html';
                if ($wp_filesystem->exists($readmeFile) && @$wp_filesystem->delete($readmeFile)) {
                    $isSuccessOperation = true;
                } else {
                    $message = __('Could not be deleted readme.html', BW_PLUGIN_SLUG);
                }
                break;
            case 'license':
                $licenseFile = ABSPATH . 'license.txt';
                if ($wp_filesystem->exists($licenseFile) && @$wp_filesystem->delete($licenseFile)) {
                    $isSuccessOperation = true;
                } else {
                    $message = __('Could not be deleted license.txt', BW_PLUGIN_SLUG);
                }
                break;
            case 'debug_log':
                $debugLogFile = ABSPATH . 'wp-content/debug.log';
                if ($wp_filesystem->exists($debugLogFile) && @$wp_filesystem->delete($debugLogFile)) {
                    $isSuccessOperation = true;
                } else {
                    $message = __('Could not be deleted debug.log', BW_PLUGIN_SLUG);
                }
                break;
        }

        if ($isSuccessOperation) {
            $response = ['result' => 'successful'];
        } else {
            $response = ['result' => 'error', 'message' => $message];
        }

        wp_send_json($response);
    }

    /**
     * AJAX action
     */
    public function change_debug_log()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $enable = isset($_POST['enable']) && $_POST['enable'] === 'true';

        $wp_filesystem = FileSystemHelper::get_filesystem();
        $configFile = ABSPATH . 'wp-config.php';

        if ($wp_filesystem->exists($configFile) && $wp_filesystem->is_writable($configFile)) {
            $content = $wp_filesystem->get_contents($configFile);

            if ($enable) {
                $find = 'false';
                $replace = 'true';
            } else {
                $find = 'true';
                $replace = 'false';
            }

            $commentedRegex = "/(?:\/\/|\/\*+|\*+)\s*define\s*\(\s*[\"']WP_DEBUG_LOG[\"']\s*,\s*(true|false)\s*\s*\)\s*;.*?(?:\*\/)?/s";
            $isExists = "/define\s*\(\s*[\"']WP_DEBUG_LOG[\"']\s*,\s*(true|false)\s*\s*\)\s*;/";
            if (preg_match($commentedRegex, $content) || !preg_match($isExists, $content)) {
                $response = ['result' => 'error', 'message' => __('The WP_DEBUG_LOG setting is missing in config.php', BW_PLUGIN_SLUG)];
            } else {
                $content = preg_replace(
                    "/define\s*\(\s*[\"']WP_DEBUG_LOG[\"']\s*,\s*{$find}\s*\s*\)\s*;/",
                    "define(\"WP_DEBUG_LOG\", {$replace});",
                    $content
                );

                if ($wp_filesystem->put_contents($configFile, $content)) {
                    $response = ['result' => 'successful'];
                } else {
                    $response = ['result' => 'error', 'message' => __('Could not be updated config.php', BW_PLUGIN_SLUG)];
                }
            }
            wp_send_json($response);
        } else {
            wp_send_json(['result' => 'error', 'message' => __('Could not be changed config.php', BW_PLUGIN_SLUG)]);
        }
    }

    /**
     * AJAX action
     */
    public function change_rest_api()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $mainHelper = new MainHelper();
        $enable = isset($_POST['enable']) && $_POST['enable'] === 'true';

        $settingsModel = new SettingsModel();
        if ($settingsModel->update_by_name($settingsModel->rest_api_enabled, [
            $settingsModel->settings_structure->value_number => ($enable) ? SettingsModel::REST_API_ENABLED : SettingsModel::REST_API_DISABLED
        ])) {
            $currentIsEnabled = $mainHelper->checkIsEnabledRestApi();
            if ($currentIsEnabled == $enable) {
                $response = ['result' => 'successful'];

                wp_send_json($response);
            } else {
                wp_send_json(['result' => 'error', 'message' => __('Could not be changed REST API status', BW_PLUGIN_SLUG)]);
            }
        } else {
            wp_send_json(['result' => 'error', 'message' => __('Could not be changed REST API status', BW_PLUGIN_SLUG)]);
        }
    }

    /**
     * AJAX action
     */
    public function change_xml_rpc()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $mainHelper = new MainHelper();
        $enable = isset($_POST['enable']) && $_POST['enable'] === 'true';

        $settingsModel = new SettingsModel();
        if ($settingsModel->update_by_name($settingsModel->xml_rpc_enabled, [
            $settingsModel->settings_structure->value_number => ($enable) ? SettingsModel::XML_RPC_ENABLED : SettingsModel::XML_RPC_DISABLED
        ])) {
            $currentIsEnabled = $mainHelper->checkIsEnabledXmlRpc();
            if ($currentIsEnabled == $enable) {
                $response = ['result' => 'successful'];

                wp_send_json($response);
            } else {
                wp_send_json(['result' => 'error', 'message' => __('Could not be changed XML-RPC status', BW_PLUGIN_SLUG)]);
            }
        } else {
            wp_send_json(['result' => 'error', 'message' => __('Could not be changed XML-RPC status', BW_PLUGIN_SLUG)]);
        }
    }

    /**
     * AJAX action
     */
    public function change_rss()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $mainHelper = new MainHelper();
        $enable = isset($_POST['enable']) && $_POST['enable'] === 'true';

        $settingsModel = new SettingsModel();
        if ($settingsModel->update_by_name($settingsModel->rss_enabled, [
            $settingsModel->settings_structure->value_number => ($enable) ? SettingsModel::RSS_ENABLED : SettingsModel::RSS_DISABLED
        ])) {
            if ($enable) {
                remove_filter('feed_links_show_posts_feed', '__return_false');
                remove_filter('feed_links_show_comments_feed', '__return_false');
            } else {
                add_filter('feed_links_show_posts_feed', '__return_false');
                add_filter('feed_links_show_comments_feed', '__return_false');
            }

            $currentIsEnabled = $mainHelper->checkIsEnabledRss();
            if ($currentIsEnabled == $enable) {
                $response = ['result' => 'successful'];

                wp_send_json($response);
            } else {
                wp_send_json(['result' => 'error', 'message' => __('Could not be changed RSS status', BW_PLUGIN_SLUG)]);
            }
        } else {
            wp_send_json(['result' => 'error', 'message' => __('Could not be changed RSS status', BW_PLUGIN_SLUG)]);
        }
    }

    /**
     * AJAX action
     */
    public function delete_unused_theme()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $themeSlug = sanitize_text_field($_POST['slug'] ?? '');

        if (empty($themeSlug)) {
            wp_send_json(['result' => 'error', 'message' => __('Theme slug param is missing', BW_PLUGIN_SLUG)]);
        }

        $allThemes = wp_get_themes();
        $activeTheme = wp_get_theme();

        if (!isset($allThemes[$themeSlug])) {
            wp_send_json(['result' => 'error', 'message' => __('Theme not found', BW_PLUGIN_SLUG)]);
        } elseif ($themeSlug === $activeTheme->get_stylesheet() || $themeSlug === $activeTheme->get_template()) {
            wp_send_json(['result' => 'error', 'message' => __('Cannot delete the active theme', BW_PLUGIN_SLUG)]);
        }

        $mainHelper = new MainHelper();
        $wp_filesystem = FileSystemHelper::get_filesystem();
        $themeDir = get_theme_root() . '/' . $themeSlug;

        if ($wp_filesystem->is_dir($themeDir)) {
            if ($mainHelper->deleteDirectory($themeDir)) {
                wp_send_json(['result' => 'successful']);
            } else {
                wp_send_json(['result' => 'error', 'message' => __('Could not be deleted theme directory', BW_PLUGIN_SLUG)]);
            }
        } else {
            wp_send_json(['result' => 'error', 'message' => __('Theme directory not found', BW_PLUGIN_SLUG)]);
        }
    }

    /**
     * AJAX action
     */
    public function delete_unused_plugin()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $pluginFile = sanitize_text_field($_POST['plugin'] ?? '');

        if (empty($pluginFile)) {
            wp_send_json(['result' => 'error', 'message' => __('Plugin file param is missing', BW_PLUGIN_SLUG)]);
        }

        $allPlugins = get_plugins();
        if (!isset($allPlugins[$pluginFile])) {
            wp_send_json(['result' => 'error', 'message' => __('Plugin not found', BW_PLUGIN_SLUG)]);
        }

        $activePlugin = get_option('active_plugins', []);
        if (in_array($pluginFile, $activePlugin)) {
            wp_send_json(['result' => 'error', 'message' => __('Cannot delete an active plugin', BW_PLUGIN_SLUG)]);
        }

        $result = delete_plugins([$pluginFile]);

        if (is_wp_error($result)) {
            wp_send_json(['result' => 'error', 'message' => __('Could not delete the plugin: ', BW_PLUGIN_SLUG) . $result->get_error_message()]);
        } else {
            wp_send_json(['result' => 'successful']);
        }
    }

    /**
     * AJAX action
     */
    public function delete_hidden_plugin()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $pluginFile = sanitize_text_field($_POST['plugin'] ?? '');

        if (empty($pluginFile)) {
            wp_send_json(['result' => 'error', 'message' => __('Plugin file param is missing', BW_PLUGIN_SLUG)]);
        }

        $allPlugins = get_plugins();
        if (!isset($allPlugins[$pluginFile])) {
            wp_send_json(['result' => 'error', 'message' => __('Plugin not found', BW_PLUGIN_SLUG)]);
        }

        if (is_plugin_active($pluginFile)) {
            deactivate_plugins($pluginFile);
        }

        $result = delete_plugins([$pluginFile]);

        if (is_wp_error($result)) {
            wp_send_json(['result' => 'error', 'message' => __('Could not delete the plugin: ', BW_PLUGIN_SLUG) . $result->get_error_message()]);
        } else {
            wp_send_json(['result' => 'successful']);
        }
    }
}
