<?php
namespace betawall\controllers;

use betawall\components\NavManager;
use betawall\components\ViewManager;
use betawall\crud\FirewallBlacklistTableData;
use betawall\crud\FirewallWhitelistTableData;
use betawall\db\models\FirewallBlacklistModel;
use betawall\db\models\FirewallWhitelistModel;
use betawall\db\models\SettingsModel;
use betawall\helpers\BinaryHelper;
use betawall\helpers\FirewallHelper;
use betawall\helpers\MainHelper;
use betawall\helpers\MuHelper;
use betawall\helpers\TasksHelper;

class FirewallController
{

    const VIEWS_DIR_NAME = 'firewall';

    public function index()
    {
        $firewallBlacklistTableData = new FirewallBlacklistTableData();
        $firewallBlacklistTableData->prepare_items();

        $firewallWhitelistTableData = new FirewallWhitelistTableData();
        $firewallWhitelistTableData->prepare_items();

        $settingsModel = new SettingsModel();
        $unblockKey = $settingsModel->get_by_name($settingsModel->unblock_key)->value_text ?? '';
        $unblockLink = (!empty($unblockKey)) ? MainHelper::getUnblockLink($unblockKey) : '';

        $binaryHelper = BinaryHelper::getInstance();

        $firewallBlacklistModel = new FirewallBlacklistModel();
        $firewallBlacklistCount = $firewallBlacklistModel->get_total_count();
        $firewallBlacklistCacheIpv4Count = $binaryHelper->getCountIpsInFile(BinaryHelper::PROTOCOL_IPV4);
        $firewallBlacklistCacheIpv6Count = $binaryHelper->getCountIpsInFile(BinaryHelper::PROTOCOL_IPV6);
        $firewallBlacklistCacheSumCount = $firewallBlacklistCacheIpv4Count + $firewallBlacklistCacheIpv6Count;

        $firewallWhitelistModel = new FirewallWhitelistModel();
        $firewallWhitelistCount = $firewallWhitelistModel->get_total_count();
        $firewallWhitelistCacheIpv4Count = $binaryHelper->getCountIpsInFile(BinaryHelper::PROTOCOL_IPV4, true);
        $firewallWhitelistCacheIpv6Count = $binaryHelper->getCountIpsInFile(BinaryHelper::PROTOCOL_IPV6, true);
        $firewallWhitelistCacheSumCount = $firewallWhitelistCacheIpv4Count + $firewallWhitelistCacheIpv6Count;

        $isFirewallCacheSynchronized = ($firewallBlacklistCount == $firewallBlacklistCacheSumCount) && ($firewallWhitelistCount == $firewallWhitelistCacheSumCount);

        $navManager = new NavManager();
        $viewManager = new ViewManager();
        $viewManager->render(self::VIEWS_DIR_NAME, '_index', [
            'firewall_blacklist_table_data' => $firewallBlacklistTableData,
            'firewall_whitelist_table_data' => $firewallWhitelistTableData,
            'unblock_link' => $unblockLink,
            'is_firewall_cache_synchronized' => $isFirewallCacheSynchronized,
            'nav_manager' => $navManager,
        ]);
    }

    /**
     * AJAX action
     */
    public function move_to_whitelist()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $itemId = $_POST['id'] ?? 0;
        $itemId = (int)$itemId;

        if ($itemId <= 0) {
            $response = ['result' => 'error', 'message' => __('Invalid ID', BW_PLUGIN_SLUG)];
        } else {
            $firewallBlacklistModel = new FirewallBlacklistModel();
            $item = $firewallBlacklistModel->get_by_id($itemId);

            if (!empty($item)) {
                $binaryHelper = BinaryHelper::getInstance();
                if (!$binaryHelper->moveFromFirewallBlacklistToWhitelist($item->ip)) {
                    $response = ['result' => 'error', 'message' => __('Operation error', BW_PLUGIN_SLUG)];
                } else {
                    $response = ['result' => 'successful'];
                }
            } else {
                $response = ['result' => 'error', 'message' => __('IP not found in blacklist', BW_PLUGIN_SLUG)];
            }
        }
        wp_send_json($response);
    }

    /**
     * AJAX action
     */
    public function delete_from_blacklist()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $itemId = $_POST['id'] ?? 0;
        $itemId = (int)$itemId;

        if ($itemId <= 0) {
            $response = ['result' => 'error', 'message' => __('Invalid ID', BW_PLUGIN_SLUG)];
        } else {
            $firewallBlacklistModel = new FirewallBlacklistModel();
            $item = $firewallBlacklistModel->get_by_id($itemId);

            if (!empty($item)) {
                $binaryHelper = BinaryHelper::getInstance();
                if (!$binaryHelper->deleteFromFirewallList($item->ip)) {
                    $response = ['result' => 'error', 'message' => __('Operation error', BW_PLUGIN_SLUG)];
                } else {
                    $response = ['result' => 'successful'];
                }
            } else {
                $response = ['result' => 'error', 'message' => __('IP not found in blacklist', BW_PLUGIN_SLUG)];
            }
        }

        wp_send_json($response);
    }

    /**
     * AJAX action
     */
    public function delete_from_whitelist()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $itemId = $_POST['id'] ?? 0;
        $itemId = (int)$itemId;

        if ($itemId <= 0) {
            $response = ['result' => 'error', 'message' => __('Invalid ID', BW_PLUGIN_SLUG)];
        } else {
            $firewallWhitelistModel = new FirewallWhitelistModel();
            $item = $firewallWhitelistModel->get_by_id($itemId);

            if (!empty($item)) {
                $binaryHelper = BinaryHelper::getInstance();
                if (!$binaryHelper->deleteFromFirewallList($item->ip, true)) {
                    $response = ['result' => 'error', 'message' => __('Operation error', BW_PLUGIN_SLUG)];
                } else {
                    $allowedIpsDisabled = FirewallHelper::getInstance()->checkAndDisableAllowedIps();
                    $response = [
                        'result'               => 'successful',
                        'allowed_ips_disabled' => $allowedIpsDisabled,
                        'message'              => __('Allowed IPs is disabled', BW_PLUGIN_SLUG),
                    ];
                }
            } else {
                $response = ['result' => 'error', 'message' => __('IP not found in whitelist', BW_PLUGIN_SLUG)];
            }
        }

        wp_send_json($response);
    }

    /**
     * AJAX action
     */
    public function add_to_whitelist()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $ip = isset($_POST['ip']) ? sanitize_text_field($_POST['ip']) : '';

        if (empty($ip)) {
            $response = ['result' => 'error', 'message' => __('IP Address is blank', BW_PLUGIN_SLUG)];
        } else if (!filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4) && !filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            $response = ['result' => 'error', 'message' => __('Invalid IP Address', BW_PLUGIN_SLUG)];
        } else {
            $binaryHelper = BinaryHelper::getInstance();
            if ($binaryHelper->addToFirewallWhitelist($ip)) {
                $response = ['result' => 'successful'];
            } else {
                $response = ['result' => 'error', 'message' => __('Could not be added IP Address to whitelist', BW_PLUGIN_SLUG)];
            }
        }

        wp_send_json($response);
    }

    /**
     * AJAX action
     */
    public function change_firewall_protection()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $binaryHelper = BinaryHelper::getInstance();
        if ($binaryHelper->searchInPermanentList(BinaryHelper::S) === false) {
            $response = ['result' => 'forbidden', 'message' => __('Could not disable all firewall settings', BW_PLUGIN_SLUG)];
        } else {
            $enable = isset($_POST['enable']) && $_POST['enable'] === 'true';
            $settingsModel = new SettingsModel();

            if ($enable) {
                $result = $settingsModel->enable_all_protection_settings();
                $errorMessage = __('Could not enable all firewall settings', BW_PLUGIN_SLUG);
            } else {
                $result = $settingsModel->disable_all_protection_settings();
                $errorMessage = __('Could not disable all firewall settings', BW_PLUGIN_SLUG);
            }

            if ($result !== false) {
                $mainHelper = new MainHelper();
                $currentProtectionInfo = $mainHelper->getCurrentProtectionInfo();
                $response = ['result' => 'successful'];
                $response = array_merge($response, $currentProtectionInfo);
            } else {
                $response = ['result' => 'error', 'message' => $errorMessage];
            }
        }

        wp_send_json($response);
    }

    /**
     * AJAX action
     */
    public function change_mu_plugin()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $enable = isset($_POST['enable']) && $_POST['enable'] === 'true';

        $muHelper = new MuHelper();
        if ($enable) {
            $process = $muHelper->changeMuPluginMode(true);
        } else {
            $process = $muHelper->changeMuPluginMode(false);
        }

        if ($process['result']) {
            $response = ['result' => 'successful'];
        } else {
            $response = ['result' => 'error', 'message' => __('Could not be changed MU Plugin status', BW_PLUGIN_SLUG)];
        }

        wp_send_json($response);
    }

    /**
     * AJAX action
     */
    public function get_db_update()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $tasksHelper = new TasksHelper();
        $tasksHelper->malware_get_update();

        $mainHelper = new MainHelper();
        $lastDatabaseUpdate = $mainHelper->getDateTimeLastDatabaseUpdate();

        wp_send_json(['result' => 'successful', 'last_database_update' => $lastDatabaseUpdate]);
    }

    public function settings()
    {
        $viewManager = new ViewManager();
        $binaryHelper = BinaryHelper::getInstance();

        $settingsModel = new SettingsModel();
        $firewallAllSettings = $settingsModel->get_firewall_protection_all_settings();

        $firewallAllSettingsRender = [];

        if (!empty($firewallAllSettings)) {
            $mapping = [
                $settingsModel->firewall_notfound_5m_limit => [
                    'group' => $settingsModel->firewall_notfound_enabled,
                    'limit_key' => '5m_limit',
                    'class' => 'firewall_notfound_5m_limit',
                ],
                $settingsModel->firewall_notfound_24h_limit => [
                    'group' => $settingsModel->firewall_notfound_enabled,
                    'limit_key' => '24h_limit',
                    'class' => 'firewall_notfound_24h_limit',
                ],
                $settingsModel->firewall_notfound_enabled => [
                    'is_enabled' => $settingsModel->firewall_notfound_enabled,
                    'module_id' => '404-request',
                    'module_info' => __('404 Request Flood is a type of DDoS attack in which attackers send a large number of HTTP requests that return a 404 Not Found server response code.', BW_PLUGIN_SLUG),
                    'module_name' => __('404 Request Flood', BW_PLUGIN_SLUG),
                ],


                $settingsModel->firewall_failed_login_5m_limit => [
                    'group' => $settingsModel->firewall_failed_login_enabled,
                    'limit_key' => '5m_limit',
                    'class' => 'firewall_failed_login_5m_limit',
                ],
                $settingsModel->firewall_failed_login_24h_limit => [
                    'group' => $settingsModel->firewall_failed_login_enabled,
                    'limit_key' => '24h_limit',
                    'class' => 'firewall_failed_login_24h_limit',
                ],
                $settingsModel->firewall_failed_login_enabled => [
                    'is_enabled' => $settingsModel->firewall_failed_login_enabled,
                    'module_id' => 'login-request',
                    'module_info' => __('Admin Panel Brute Force is a type of attack targeting the admin panel of a website or application, where attackers try to guess the login and password by mass traversal of possible combinations.', BW_PLUGIN_SLUG),
                    'module_name' => __('Admin Panel Brute Force', BW_PLUGIN_SLUG),
                ],


                $settingsModel->firewall_sql_injection_5m_limit => [
                    'group' => $settingsModel->firewall_sql_injection_enabled,
                    'limit_key' => '5m_limit',
                    'class' => 'firewall_sql_injection_5m_limit',
                ],
                $settingsModel->firewall_sql_injection_24h_limit => [
                    'group' => $settingsModel->firewall_sql_injection_enabled,
                    'limit_key' => '24h_limit',
                    'class' => 'firewall_sql_injection_24h_limit',
                ],
                $settingsModel->firewall_sql_injection_enabled => [
                    'is_enabled' => $settingsModel->firewall_sql_injection_enabled,
                    'module_id' => 'sql-request',
                    'module_info' => __('SQL Injection Attacks are a type of cyberattack in which attackers inject malicious SQL code into data input fields or database queries to gain unauthorized access to or modify data.', BW_PLUGIN_SLUG),
                    'module_name' => __('SQL Injection Attacks', BW_PLUGIN_SLUG),
                ],


                $settingsModel->firewall_code_execution_5m_limit => [
                    'group' => $settingsModel->firewall_code_execution_enabled,
                    'limit_key' => '5m_limit',
                    'class' => 'firewall_code_execution_5m_limit',
                ],
                $settingsModel->firewall_code_execution_24h_limit => [
                    'group' => $settingsModel->firewall_code_execution_enabled,
                    'limit_key' => '24h_limit',
                    'class' => 'firewall_code_execution_24h_limit',
                ],
                $settingsModel->firewall_code_execution_enabled => [
                    'is_enabled' => $settingsModel->firewall_code_execution_enabled,
                    'module_id' => 'arbitrary-request',
                    'module_info' => __('Arbitrary Code Execution (ACE) is a type of cyberattack in which attackers gain the ability to execute any code on a target system or server. This is one of the most dangerous vulnerabilities because an attacker can take complete control of the system if he manages to execute the code.', BW_PLUGIN_SLUG),
                    'module_name' => __('Arbitrary Code Execution', BW_PLUGIN_SLUG),
                ],


                $settingsModel->firewall_xss_5m_limit => [
                    'group' => $settingsModel->firewall_xss_enabled,
                    'limit_key' => '5m_limit',
                    'class' => 'firewall_xss_5m_limit',
                ],
                $settingsModel->firewall_xss_24h_limit => [
                    'group' => $settingsModel->firewall_xss_enabled,
                    'limit_key' => '24h_limit',
                    'class' => 'firewall_xss_24h_limit',
                ],
                $settingsModel->firewall_xss_enabled => [
                    'is_enabled' => $settingsModel->firewall_xss_enabled,
                    'module_id' => 'xss-request',
                    'module_info' => __('XSS (Cross-Site Scripting) is a type of attack in which an attacker inserts malicious script (usually JavaScript) into a web page, which is then executed on the user\'s end when the user interacts with the site. The purpose of such an attack is to manipulate or steal user data, as well as to force the browser to execute malicious code.', BW_PLUGIN_SLUG),
                    'module_name' => __('XSS Attacks', BW_PLUGIN_SLUG),
                ],


                $settingsModel->firewall_rfi_5m_limit => [
                    'group' => $settingsModel->firewall_rfi_enabled,
                    'limit_key' => '5m_limit',
                    'class' => 'firewall_rfi_5m_limit',
                ],
                $settingsModel->firewall_rfi_24h_limit => [
                    'group' => $settingsModel->firewall_rfi_enabled,
                    'limit_key' => '24h_limit',
                    'class' => 'firewall_rfi_24h_limit',
                ],
                $settingsModel->firewall_rfi_enabled => [
                    'is_enabled' => $settingsModel->firewall_rfi_enabled,
                    'module_id' => 'rfi-request',
                    'module_info' => __('RFI (Remote File Inclusion) is a type of vulnerability in web applications where an attacker can include a file located on a remote server in the execution of the application. This type of attack is typically used to execute malicious code on a server or obtain sensitive information.', BW_PLUGIN_SLUG),
                    'module_name' => __('RFI Attacks', BW_PLUGIN_SLUG),
                ],


                $settingsModel->firewall_suspicious_file_uploads_5m_limit => [
                    'group' => $settingsModel->firewall_suspicious_file_uploads_enabled,
                    'limit_key' => '5m_limit',
                    'class' => 'firewall_suspicious_file_uploads_5m_limit',
                ],
                $settingsModel->firewall_suspicious_file_uploads_24h_limit => [
                    'group' => $settingsModel->firewall_suspicious_file_uploads_enabled,
                    'limit_key' => '24h_limit',
                    'class' => 'firewall_suspicious_file_uploads_24h_limit',
                ],
                $settingsModel->firewall_suspicious_file_uploads_enabled => [
                    'is_enabled' => $settingsModel->firewall_suspicious_file_uploads_enabled,
                    'module_id' => 'file-request',
                    'module_info' => __('Suspicious File Uploads (suspicious file uploads) is a type of vulnerability or dangerous behavior where files are uploaded to a server or website that could be malicious or used for an attack. This can include downloading executable files, scripts, viruses, or even files that masquerade as safe documents, but may actually pose a threat to the system.', BW_PLUGIN_SLUG),
                    'module_name' => __('Suspicious File Uploads', BW_PLUGIN_SLUG),
                ],
            ];

            foreach ($firewallAllSettings as $setting) {
                $groupKey = $mapping[$setting->name] ?? null;

                if ($groupKey) {
                    if (isset($groupKey['is_enabled'])) {
                        if (!isset($firewallAllSettingsRender[$groupKey['is_enabled']])) {
                            $firewallAllSettingsRender[$groupKey['is_enabled']] = [
                                'is_enabled' => ($binaryHelper->searchInPermanentList(BinaryHelper::S) !== false) ? $setting->value_number : 0,
                                'module_id' => $groupKey['module_id'],
                                'module_info' => $groupKey['module_info'],
                                'module_name' => $groupKey['module_name'],
                            ];
                        } else {
                            $firewallAllSettingsRender[$groupKey['is_enabled']]['is_enabled'] = ($binaryHelper->searchInPermanentList(BinaryHelper::S) !== false) ? $setting->value_number : 0;
                        }
                    } elseif (isset($groupKey['group'], $groupKey['limit_key'])) {
                        if (!isset($firewallAllSettingsRender[$groupKey['group']])) {
                            $firewallAllSettingsRender[$groupKey['group']] = [];
                        }
                        $firewallAllSettingsRender[$groupKey['group']][$groupKey['limit_key']] = $setting->value_number;

                        if (!isset($firewallAllSettingsRender[$groupKey['group']]['5m_limit_class'])) {
                            $firewallAllSettingsRender[$groupKey['group']]['5m_limit_class'] = $groupKey['class'];
                        } else {
                            $firewallAllSettingsRender[$groupKey['group']]['24h_limit_class'] = $groupKey['class'];
                        }
                    }
                }

                if ($setting->name === $settingsModel->firewall_rfi_external_urls_enabled) {
                    if (!isset($firewallAllSettingsRender[$settingsModel->firewall_rfi_enabled]['submodules'])) {
                        $firewallAllSettingsRender[$settingsModel->firewall_rfi_enabled]['submodules'] = [];
                    }

                    $firewallAllSettingsRender[$settingsModel->firewall_rfi_enabled]['submodules'][] = [
                        'is_enabled' => ($binaryHelper->searchInPermanentList(BinaryHelper::S) !== false) ? $setting->value_number : 0,
                        'module_id' => 'rfi-external-urls-request',
                        'module_name' => __('Block external URLs', BW_PLUGIN_SLUG),
                    ];
                }
            }
        }

        $settings = FirewallHelper::getInstance()->getDefaultsProtectionSettings();

        $muHelper = new MuHelper();
        $muPluginStatus = $muHelper->isEnabledMuPluginMode();

        $g2faIsEnabled = ($binaryHelper->searchInPermanentList(BinaryHelper::S) === false) ? SettingsModel::G2FA_DISABLED : $settingsModel->get_by_name($settingsModel->g2fa_enabled)->value_number ?? SettingsModel::G2FA_DISABLED;

        $viewManager->render(self::VIEWS_DIR_NAME, '_settings', [
            'firewall_all_settings'    => $firewallAllSettingsRender,
            'is_enabled_mu_plugin'     => $muPluginStatus,
            'recaptcha'                => ($binaryHelper->searchInPermanentList(BinaryHelper::S) === false) ? SettingsModel::RECAPTCHA_DISABLED : $settingsModel->get_by_name($settingsModel->recaptcha_enabled)->value_number ?? SettingsModel::RECAPTCHA_DISABLED,
            'recaptcha_secret_key'     => $settingsModel->get_by_name($settingsModel->recaptcha_secret_key)->value_text ?? '',
            'recaptcha_site_key'       => $settingsModel->get_by_name($settingsModel->recaptcha_site_key)->value_text ?? '',
            'g2fa'                     => $g2faIsEnabled,
            'g2faRolesSelectHtmlBlock' => FirewallHelper::getInstance()->g2faRolesSelectHtmlBlock(),
            'g2faUsersListHtmlBlock'   => FirewallHelper::getInstance()->g2faUsersListHtmlBlock($g2faIsEnabled),
            'default_settings_threat_monitor' => json_encode($settings),
        ]);
    }

    /**
     * AJAX action
     */
    public function reset_defaults_protection()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $settingsModel = new SettingsModel();
        $binaryHelper = BinaryHelper::getInstance();

        $success = true;
        $successResponse = ['result' => 'successful'];
        $errorResponse = ['result' => 'error', 'message' => __('Could not reset default settings', BW_PLUGIN_SLUG)];

        if ($binaryHelper->searchInPermanentList(BinaryHelper::S) === false) {
            $success = false;
            $errorResponse = ['result' => 'forbidden', 'message' => __('Could not reset default settings', BW_PLUGIN_SLUG)];
        } else {
            $settings = FirewallHelper::getInstance()->getDefaultsProtectionSettings();

            foreach ($settings as $key => $value) {
                if (!$settingsModel->update_by_name($key, [$settingsModel->settings_structure->value_number => $value])) {
                    $success = false;
                    break;
                }
            }
        }

        $response = ($success) ? $successResponse : $errorResponse;
        wp_send_json($response);
    }

    /**
     * AJAX action
     */
    public function save_protection_settings()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $binaryHelper = BinaryHelper::getInstance();
        if ($binaryHelper->searchInPermanentList(BinaryHelper::S) === false) {
            wp_send_json(['result' => 'forbidden', 'message' => __('Changes could not be saved', BW_PLUGIN_SLUG)]);
        }

        $settingsModel = new SettingsModel();
        $mainHelper = new MainHelper();

        $modules = [
            $settingsModel->firewall_notfound_enabled,
            $settingsModel->firewall_failed_login_enabled,
            $settingsModel->firewall_sql_injection_enabled,
            $settingsModel->firewall_code_execution_enabled,
            $settingsModel->firewall_xss_enabled,
            $settingsModel->firewall_rfi_enabled,
            $settingsModel->firewall_rfi_external_urls_enabled,
            $settingsModel->firewall_suspicious_file_uploads_enabled,
        ];

        $modulesNeedEnable = [];
        $modulesNeedDisable = [];
        $dataEnabled = $_POST['data_enabled'] ?? [];

        foreach ($modules as $module) {
            $moduleIsEnabled = isset($dataEnabled[$module]) && $dataEnabled[$module] === 'true';
            if ($moduleIsEnabled) {
                $modulesNeedEnable[] = $module;
            } else {
                $modulesNeedDisable[] = $module;
            }
        }

        if (!empty($modulesNeedEnable)) {
            $settingsModel->enable_protection_settings($modulesNeedEnable);
        }

        if (!empty($modulesNeedDisable)) {
            $settingsModel->disable_protection_settings($modulesNeedDisable);
        }

        $requestLimits = [
            [
                'name' => $settingsModel->firewall_notfound_5m_limit,
                'selector' => '#bw-404-request-5m-input',
                'value' => 0
            ],
            [
                'name' => $settingsModel->firewall_notfound_24h_limit,
                'selector' => '#bw-404-request-24h-input',
                'value' => 0
            ],
            [
                'name' =>$settingsModel->firewall_failed_login_5m_limit,
                'selector' => '#bw-login-request-5m-input',
                'value' => 0
            ],
            [
                'name' => $settingsModel->firewall_failed_login_24h_limit,
                'selector' => '#bw-login-request-24h-input',
                'value' => 0
            ],
            [
                'name' => $settingsModel->firewall_sql_injection_5m_limit,
                'selector' => '#bw-sql-request-5m-input',
                'value' => 0
            ],
            [
                'name' => $settingsModel->firewall_sql_injection_24h_limit,
                'selector' => '#bw-sql-request-24h-input',
                'value' => 0
            ],
            [
                'name' => $settingsModel->firewall_code_execution_5m_limit,
                'selector' => '#bw-arbitrary-request-5m-input',
                'value' => 0
            ],
            [
                'name' => $settingsModel->firewall_code_execution_24h_limit,
                'selector' => '#bw-arbitrary-request-24h-input',
                'value' => 0
            ],
            [
                'name' => $settingsModel->firewall_xss_5m_limit,
                'selector' => '#bw-xss-request-5m-input',
                'value' => 0
            ],
            [
                'name' => $settingsModel->firewall_xss_24h_limit,
                'selector' => '#bw-xss-request-24h-input',
                'value' => 0
            ],
            [
                'name' => $settingsModel->firewall_rfi_5m_limit,
                'selector' => '#bw-rfi-request-5m-input',
                'value' => 0
            ],
            [
                'name' => $settingsModel->firewall_rfi_24h_limit,
                'selector' => '#bw-rfi-request-24h-input',
                'value' => 0
            ],
            [
                'name' => $settingsModel->firewall_suspicious_file_uploads_5m_limit,
                'selector' => '#bw-file-request-5m-input',
                'value' => 0
            ],
            [
                'name' => $settingsModel->firewall_suspicious_file_uploads_24h_limit,
                'selector' => '#bw-file-request-24h-input',
                'value' => 0
            ]
        ];

        $errors = [];
        $dataLimits = $_POST['data_limits'] ?? [];

        foreach ($requestLimits as $key => $field) {
            $value = isset($dataLimits[$field['name']]) ? sanitize_text_field($dataLimits[$field['name']]) : null;

            if ($value === null || !$mainHelper->validateLimitValue($value)) {
                $errors[] = [
                    'field' => $field['name'],
                    'selector' => $field['selector'],
                ];
            } else {
                $requestLimits[$key]['value'] = $value;
            }
        }

        if (!empty($errors)) {
            wp_send_json(['result' => 'error', 'errors' => $errors]);
        } else {
            $success = true;
            foreach ($requestLimits as $field) {
                if (!$settingsModel->update_by_name($field['name'], [$settingsModel->settings_structure->value_number => $field['value']])) {
                    $success = false;
                    break;
                }
            }

            if ($success !== false) {
                $successResponse = ['result' => 'successful'];
                $mainHelper = new MainHelper();
                $currentProtectionInfo = $mainHelper->getCurrentProtectionInfo();
                $response = array_merge($successResponse, $currentProtectionInfo);

                wp_send_json($response);
            } else {
                wp_send_json(['result' => 'error', 'message' => __('Changes could not be saved', BW_PLUGIN_SLUG)]);
            }
        }
    }

    /**
     * AJAX action
     */
    function cache_sync()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $binaryHelper = BinaryHelper::getInstance();
        $binaryHelper->initSysFiles();

        $firewallBlacklistModel = new FirewallBlacklistModel();
        $firewallBlacklistCount = $firewallBlacklistModel->get_total_count();
        $firewallBlacklistCacheIpv4Count = $binaryHelper->getCountIpsInFile(BinaryHelper::PROTOCOL_IPV4);
        $firewallBlacklistCacheIpv6Count = $binaryHelper->getCountIpsInFile(BinaryHelper::PROTOCOL_IPV6);
        $firewallBlacklistCacheSumCount = $firewallBlacklistCacheIpv4Count + $firewallBlacklistCacheIpv6Count;


        $firewallWhitelistModel = new FirewallWhitelistModel();
        $firewallWhitelistCount = $firewallWhitelistModel->get_total_count();
        $firewallWhitelistCacheIpv4Count = $binaryHelper->getCountIpsInFile(BinaryHelper::PROTOCOL_IPV4, true);
        $firewallWhitelistCacheIpv6Count = $binaryHelper->getCountIpsInFile(BinaryHelper::PROTOCOL_IPV6, true);
        $firewallWhitelistCacheSumCount = $firewallWhitelistCacheIpv4Count + $firewallWhitelistCacheIpv6Count;

        $isFirewallCacheSynchronized = ($firewallBlacklistCount == $firewallBlacklistCacheSumCount) && ($firewallWhitelistCount == $firewallWhitelistCacheSumCount);

        if ($isFirewallCacheSynchronized) {
            $result = [
                'result' => 'successful',
                'html' =>
                    '<div class="bw-general-scan-settings-value-item bw-state green bw-item-active">' .  __('Synchronized', BW_PLUGIN_SLUG) . '</div>',
            ];
        } else {
            $result = [
                'result' => 'error',
                'message' => __('Synchronization error', BW_PLUGIN_SLUG),
            ];
        }

        wp_send_json($result);
    }

    /**
     * AJAX action
     */
    public function save_recapthca_settings()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $errorMessage = __('Could not save reCAPTCHA settings', BW_PLUGIN_SLUG);

        $binaryHelper = BinaryHelper::getInstance();
        if ($binaryHelper->searchInPermanentList(BinaryHelper::S) === false) {
            $response = ['result' => 'forbidden', 'message' => $errorMessage];
        } else {
            $recaptcha = (int) (isset($_POST['recaptcha']) && $_POST['recaptcha']);
            $recaptchaSiteKey = isset($_POST['recaptcha_site_key']) && $_POST['recaptcha_site_key'] ? sanitize_text_field($_POST['recaptcha_site_key']) : '';
            $recaptchaSecretKey = isset($_POST['recaptcha_secret_key']) && $_POST['recaptcha_secret_key'] ? sanitize_text_field($_POST['recaptcha_secret_key']) : '';
            $unblockKey = isset($_POST['unblock_key']) && $_POST['unblock_key'] ? sanitize_text_field($_POST['unblock_key']) : '';

            $settingsModel = new SettingsModel();

            $resultRecaptcha = $settingsModel
                ->update_by_name($settingsModel->recaptcha_enabled, [$settingsModel->settings_structure->value_number => $recaptcha]);

            // case with !$unblockKey; can be from routeGuest() request only
            if (!$unblockKey) {
                $resultRecaptchaSiteKey = $settingsModel
                    ->update_by_name($settingsModel->recaptcha_site_key, [$settingsModel->settings_structure->value_text => $recaptchaSiteKey]);
                $resultRecaptchaSecretKey = $settingsModel
                    ->update_by_name($settingsModel->recaptcha_secret_key, [$settingsModel->settings_structure->value_text => $recaptchaSecretKey]);
            }

            if ($unblockKey || ($resultRecaptcha && $resultRecaptchaSiteKey && $resultRecaptchaSecretKey)) {
                $response = ['result' => 'successful'];
            } else {
                $response = ['result' => 'error', 'message' => $errorMessage];
            }
        }

        wp_send_json($response);
    }

    /**
     * AJAX action
     */
    public function change_allowed_ips_status()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $binaryHelper = BinaryHelper::getInstance();
        if ($binaryHelper->searchInPermanentList(BinaryHelper::S) === false) {
            $response = ['result' => 'forbidden', 'message' => __('Could not enable Allowed IPs', BW_PLUGIN_SLUG)];
        } else {
            $firewallHelper = FirewallHelper::getInstance();
            $response = $firewallHelper->changeAllowedIpsStatus(isset($_POST['enable']) && (int)$_POST['enable'] === SettingsModel::ALLOWED_IPS_ENABLED);
        }

        wp_send_json($response);
    }

    /**
     * AJAX action
     */
    public function check_allowed_ips_status()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $firewallHelper = FirewallHelper::getInstance();
        if ($firewallHelper->isEnabledAllowedIps()) {
            $response = ['is_enabled' => true];
        } else {
            $response = ['is_enabled' => false];
        }

        wp_send_json($response);
    }

    /**
     * AJAX action
     */
    public function save_g2fa_changes()
    {
        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $binaryHelper = BinaryHelper::getInstance();
        if ($binaryHelper->searchInPermanentList(BinaryHelper::S) === false) {
            $response = ['result' => 'forbidden', 'message' => __('Could not disable g2fa', BW_PLUGIN_SLUG)];
        } else {
            $firewallHelper = FirewallHelper::getInstance();

            $status = isset($_POST['g2fa_enabled']) && (int)$_POST['g2fa_enabled'] === SettingsModel::G2FA_ENABLED;

            $response = $firewallHelper->save_g2fa_changes($status);
        }

        wp_send_json($response);
    }

    /**
     * AJAX action
     */
    public function save_g2fa_user_changes()
    {
        if (!current_user_can('manage_options')) {
            wp_die(BW_PLUGIN_NAME . " - " . __('access denied', BW_PLUGIN_SLUG));
        }

        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $binaryHelper = BinaryHelper::getInstance();
        if ($binaryHelper->searchInPermanentList(BinaryHelper::S) === false) {
            $response = ['result' => 'forbidden', 'message' => __('Could not disable g2fa', BW_PLUGIN_SLUG)];
        } else {
            $userId = (int) ($_POST['user_id'] ?? '');
            $value = (int) ($_POST['value'] ?? '');

            $firewallHelper = FirewallHelper::getInstance();
            $response = $firewallHelper->save_g2fa_user_changes($userId, $value);
            $response['result'] = 'successful';
        }

        wp_send_json($response);
    }

    /**
     * AJAX action
     */
    public function g2fa_user_list()
    {
        if (!current_user_can('manage_options')) {
            wp_die(BW_PLUGIN_NAME . " - " . __('access denied', BW_PLUGIN_SLUG));
        }

        if (!MainHelper::checkForAjaxRequestAccess()) {
            wp_die(BW_PLUGIN_NAME . " - " . __('incorrect request', BW_PLUGIN_SLUG));
        }

        $firewallHelper = FirewallHelper::getInstance();

        $status = isset($_POST['g2fa_enabled']) && (int)$_POST['g2fa_enabled'] === SettingsModel::G2FA_ENABLED;
        $userRole = sanitize_text_field($_POST['user_role'] ?? '');

        $response = [
            'result' => 'successful',
            'html'   => $firewallHelper->g2faUsersListHtmlBlock($status, $userRole),
        ];

        wp_send_json($response);
    }
}
