<?php
namespace betawall\components;

use betawall\helpers\FileSystemHelper;

class ViewManager
{

    private $data = [];

    public $navManager;

    public function __construct()
    {
        $this->navManager = new NavManager();
    }

    /**
     * Render a view with an optional layout.
     *
     * @param string $controllerName The name of the controller.
     * @param string $view The view file to render.
     * @param array $data The data to pass to the view.
     * @param string|null $layout The layout file to wrap the view (default: 'layout.php').
     */
    public function render($controllerName, $view, $data = [], $layout = 'layout.php')
    {
        $data['view_manager'] = $this;
        $data['nav_manager'] = $this->navManager;

        $this->data = $data;

        $wp_filesystem = FileSystemHelper::get_filesystem();

        $layoutPath = BW_PLUGIN_PATH . 'views/' . $layout;

        if (!$wp_filesystem->exists($layoutPath)) {
            wp_die(BW_PLUGIN_PATH . " - layout $layout not found");
        }

        $content = $this->renderPartial($controllerName, $view, $data, true);

        // Include layout with injected content
        include($layoutPath);
    }

    public function renderPartial($controllerName, $view, $data = [], $returnContent = false)
    {
        $data = $data ?: $this->data;

        $data['view_manager'] = $this;
        $data['nav_manager'] = $this->navManager;

        $wp_filesystem = FileSystemHelper::get_filesystem();

        $viewPath = BW_PLUGIN_PATH . 'views/' . $view . '.php';
        if ($controllerName) {
            $viewPath = BW_PLUGIN_PATH . 'views/' . $controllerName . '/' . $view . '.php';
        }

        if (!$wp_filesystem->exists($viewPath)) {
            wp_die(BW_PLUGIN_PATH . " - view $view not found");
        }

        // Register custom scripts
        if ($controllerName) {
            $this->register_scripts($controllerName, $view);
        }

        // Extract data to variables
        extract($data);

        // Capture view content
        ob_start();
        include($viewPath);

        if ($returnContent) {
            return ob_get_clean();
        }

        echo ob_get_clean();
    }

    /**
     * Register scripts specific to a controller and view.
     *
     * @param string $controllerName The name of the controller.
     * @param string $view The view file name.
     */
    private function register_scripts(string $controllerName, string $view)
    {
        $scripts = [
            'general_scan' => [
                '_index' => [
                    'assets/js/general_scan/index.js'
                ],
            ],
            'firewall' => [
                '_index' => [
                    'assets/js/firewall/index.js'
                ],
                '_settings' => [
                    'assets/js/firewall/settings.js',
                    'assets/js/external/jquery.lazyload.min.js',
                ],
            ],
            'core_health' => [
                '_index' => [
                    'assets/js/core_health/index.js'
                ],
            ],
        ];

        //    allow to load actual js and css, for individual privileged hosts
        $paramTimestamp = BW_PLUGIN_VERSION;
        if (defined('BW_PLUGIN_ALLOWED_HOSTS')) {
            $host = parse_url(home_url(), PHP_URL_HOST);
            $paramTimestamp = in_array($host, BW_PLUGIN_ALLOWED_HOSTS) ? time() : BW_PLUGIN_VERSION;
        }

        if (isset($scripts[$controllerName][$view])) {
            foreach ($scripts[$controllerName][$view] as $k => $script) {
                wp_enqueue_script("{$controllerName}{$view}{$k}-script", BW_PLUGIN_URL . $script, ['jquery'], $paramTimestamp, true);
            }
        }
    }

}
