<?php
namespace betawall\api\requests;

use betawall\api\ApiClient;
use betawall\api\data\MalwareGetUpdateResponseData;
use betawall\helpers\BinaryHelper;
use betawall\helpers\FileSystemHelper;

class MalwareGetUpdateRequest
{

    const ENDPOINT = '/malware/get-update';
    private $api_client;
    public $request_data;

    public function __construct($requestData, $apiKey)
    {
        $this->api_client = new ApiClient($apiKey);
        $this->request_data = $requestData;
    }

    public function sendRequest()
    {
        $response = $this->api_client->get(self::ENDPOINT, $this->request_data);
        return $this->getResponse($response);
    }

    private function getResponse($response)
    {
        if (is_wp_error($response)) {
            return new MalwareGetUpdateResponseData(false, $response->get_error_message(), [], []);
        }
        if (json_last_error() !== JSON_ERROR_NONE) {
            return new MalwareGetUpdateResponseData(false, 'Error decoding JSON response: ' . json_last_error_msg(), [], []);
        }

        $body = wp_remote_retrieve_body($response);
        $code = wp_remote_retrieve_response_code($response);
        $decodedBody = json_decode($body, true);

        if ($code === 401) {
            return new MalwareGetUpdateResponseData(false, 'Authentication failed', ['code' => 401], []);
        }

        if (!isset($decodedBody['success'], $decodedBody['message'], $decodedBody['data'], $decodedBody['errors'])) {
            return new MalwareGetUpdateResponseData(false, 'Invalid response format.', [], []);
        }

        return new MalwareGetUpdateResponseData($decodedBody['success'], $decodedBody['message'], $decodedBody['data'], $decodedBody['errors']);
    }

    public function downloadDb($url): bool
    {
        $wp_filesystem = FileSystemHelper::get_filesystem();

        if (!$wp_filesystem->is_dir(BinaryHelper::BINARY_DB_DIRECTORY)) {
            if ($wp_filesystem->is_writable(BW_PLUGIN_PATH)) {
                @$wp_filesystem->mkdir(BinaryHelper::BINARY_DB_DIRECTORY, FS_CHMOD_DIR, true);
            } else {
                return false;
            }
        }

        if (!$wp_filesystem->is_writable(BinaryHelper::BINARY_DB_DIRECTORY)) {
            return false;
        }

        $response = wp_remote_get($url, ['timeout' => 300]);
        if (is_wp_error($response)) {
            return false;
        }

        $body = wp_remote_retrieve_body($response);
        if (empty($body)) {
            return false;
        }

        if ($wp_filesystem->put_contents(BinaryHelper::BINARY_DB_DIRECTORY . '/' . BinaryHelper::BINARY_DB_UPDATE_ARCHIVE, $body)) {
            return true;
        }

        return false;
    }

}