<?php
/**
 * Plugin Name: BETAWALL
 * PLugin URI: https://betawall.pro/
 * Description: Security plugin designed to protect websites from a variety of threats
 * Author: Syntrex Digital
 * Author URI: https://syntrex-digital.com/
 * Version: 1.2.6
 * Requires PHP: 7.4
 */

if (file_exists( __DIR__ . '/own_const.php')) {
    require_once __DIR__ . '/own_const.php';
}

if (!defined('BW_PLUGIN_ALLOWED_HOSTS')) {
    define('BW_PLUGIN_ALLOWED_HOSTS', []);
}

if (!function_exists('getallheaders'))
{
	function getallheaders()
	{
		$headers = [];
		foreach ($_SERVER as $name => $value) {
			if (substr($name, 0, 5) == 'HTTP_') {
				$headers[str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($name, 5)))))] = $value;
			}
		}
		if (isset($_SERVER['CONTENT_TYPE'])) {
			$headers['Content-Type'] = $_SERVER['CONTENT_TYPE'];
		}
		if (isset($_SERVER['CONTENT_LENGTH'])) {
			$headers['Content-Length'] = $_SERVER['CONTENT_LENGTH'];
		}
		if (isset($_SERVER['CONTENT_MD5'])) {
			$headers['Content-Md5'] = $_SERVER['CONTENT_MD5'];
		}
		return $headers;
	}
}

//Constants
if (!defined('ABSPATH')) {
    exit;
}
if (!defined('BW_PLUGIN_NAME')) {
    define('BW_PLUGIN_NAME', 'BETAWALL');
}
if (!defined('BW_PLUGIN_SLUG')) {
    define('BW_PLUGIN_SLUG', 'betawall');
}
if (!defined('BW_PLUGIN_PATH')) {
    define('BW_PLUGIN_PATH', plugin_dir_path(__FILE__));
}
if (!defined('BW_PLUGIN_URL')) {
    define('BW_PLUGIN_URL', plugin_dir_url(__FILE__));
}
if (!defined('BW_PLUGIN_VERSION')) {
	$pluginData = get_file_data(BW_PLUGIN_PATH . 'betawall.php', array('Version' => 'Version'));
	$pluginVersion = $pluginData['Version'];
	define('BW_PLUGIN_VERSION', $pluginVersion);
}
if (!defined('BW_BACKUPS_DIR')) {
    define('BW_BACKUPS_DIR', BW_PLUGIN_PATH . 'backups/');
}

//Classmap
if (file_exists(__DIR__ . '/vendor/autoload.php')) {
    require_once __DIR__ . '/vendor/autoload.php';
}

//Uses
use betawall\components\Router;
use betawall\helpers\InitHelper;
use betawall\helpers\RestApiHelper;
use betawall\helpers\TasksHelper;
use betawall\helpers\FirewallHelper;
use betawall\helpers\MainHelper;
use betawall\db\models\SettingsModel;
use betawall\components\ViewManager;

//Classes
$initHelper = new InitHelper();
$cronHelper = new TasksHelper();
$restApiHelper = new RestApiHelper();
$firewallHelper = FirewallHelper::getInstance();

//System
add_action('admin_menu', 'bw_register_admin_menu');
add_filter('parent_file', 'bw_set_active_submenu');
add_action('admin_enqueue_scripts', 'bw_enqueue_admin_assets');
add_action('rest_api_init', 'bw_rest_api_init');
add_action('wp_ajax_bw_action', 'bw_handle_ajax_request');
add_action('wp_ajax_nopriv_bw_action', 'bw_handle_ajax_guest_request');

add_action('admin_head', 'bw_custom_admin_background_conditional');
add_action('admin_notices', [$initHelper, 'error_notice']);
add_action('wp_dashboard_setup', 'bw_add_dashboard_widget');

//Schedules
add_action(TasksHelper::BW_SCHEDULE_MALWARE_SEND, [$cronHelper, 'malware_send']);
add_action(TasksHelper::BW_SCHEDULE_MALWARE_GET_UPDATE, [$cronHelper, 'malware_get_update']);

//Init
register_activation_hook(__FILE__, [$initHelper, 'on_activation']);
register_deactivation_hook(__FILE__, [$initHelper, 'on_deactivation']);

//Firewall & General
add_action('init', [$firewallHelper, 'track_init']);
add_action('wp', [$firewallHelper, 'track_wp']);
add_action('wp_login_failed', [$firewallHelper, 'track_failed_login_request']);
add_filter('rest_authentication_errors', [$restApiHelper, 'rest_authentication']);
add_filter('plugin_row_meta', 'bw_plugin_meta', 10, 2);

function bw_handle_ajax_request() {
    $router = new Router();
    $router->route();
}
function bw_handle_ajax_guest_request() {
    $router = new Router();
    $router->routeGuest();
}

function bw_register_admin_menu() {
    $router = new Router();
    add_menu_page(
        BW_PLUGIN_NAME, BW_PLUGIN_NAME, 'manage_options',
        BW_PLUGIN_SLUG, [$router, 'route'], BW_PLUGIN_URL . 'assets/img/menu_icon.svg?v=2', 70
    );

    add_submenu_page(
        BW_PLUGIN_SLUG,
        __('Dashboard', BW_PLUGIN_SLUG),
        __('Dashboard', BW_PLUGIN_SLUG),
        'manage_options',
        BW_PLUGIN_SLUG,
        [$router, 'route']
    );

    $submenuItems = [
        [
            'title' => __('General scan', BW_PLUGIN_SLUG),
            'slug' => BW_PLUGIN_SLUG . '&c=general_scan&a=index',
        ],
        [
            'title' => __('Firewall', BW_PLUGIN_SLUG),
            'slug' => BW_PLUGIN_SLUG . '&c=firewall&a=index',
        ],
        [
            'title' => __('Settings', BW_PLUGIN_SLUG),
            'slug' => BW_PLUGIN_SLUG . '&c=firewall&a=settings',
        ],
    ];

    foreach ($submenuItems as $item) {
        add_submenu_page(
            BW_PLUGIN_SLUG,
            $item['title'],
            $item['title'],
            'manage_options',
            $item['slug'],
            [$router, 'route']
        );
    }
}

function bw_enqueue_admin_assets() {
    $settingsModel = new SettingsModel();
    $settingState = $settingsModel->get_by_name($settingsModel->error_states)->value_number ?? SettingsModel::ERROR_STATE_NO_ERRORS;

    //    allow to load actual js and css, for individual privileged hosts
    $paramTimestamp = BW_PLUGIN_VERSION;
    if (defined('BW_PLUGIN_ALLOWED_HOSTS')) {
        $host = parse_url(home_url(), PHP_URL_HOST);
        $paramTimestamp = in_array($host, BW_PLUGIN_ALLOWED_HOSTS) ? time() : BW_PLUGIN_VERSION;
    }

    wp_enqueue_style('bw_styles', BW_PLUGIN_URL . 'assets/css/style.css', [], $paramTimestamp);
    wp_enqueue_script('bw_scripts', BW_PLUGIN_URL . 'assets/js/main.js', ['jquery'], $paramTimestamp, true);

    wp_localize_script('bw_scripts', 'bw_data', [
        'ajaxUrl' => admin_url('admin-ajax.php'),
        'activationModal' => ($settingState == SettingsModel::ERROR_STATE_NEED_ACTIVATION_MODAL) ? 1 : 0,
    ]);

    if ($settingState == SettingsModel::ERROR_STATE_NEED_ACTIVATION_MODAL) {
        $settingsModel->update_by_name($settingsModel->error_states, [
            $settingsModel->settings_structure->value_number => SettingsModel::ERROR_STATE_NEED_ACTIVATION
        ]);
    }
}

function bw_rest_api_init() {
    $restApiHelper = new RestApiHelper();
    register_rest_route('betawall/v1', '/verification/', [
        'methods' => 'POST',
        'callback' => [$restApiHelper, 'rest_api_verification'],
        'permission_callback' => '__return_true',
    ]);
}

function bw_custom_admin_background_conditional() {
    $screen = get_current_screen();
    $target_screen_id = 'toplevel_page_' . BW_PLUGIN_SLUG;
    if ($screen && $screen->id === $target_screen_id) {
        $background_url = plugins_url('assets/img/screen_bg.webp', __FILE__);
        echo '<style>#wpwrap { background: url("' . esc_url($background_url) . '") no-repeat top center / contain; }</style>';
    }
}

add_action('admin_footer', function() {
    $settingsModel = new SettingsModel();
    $unblockKey = $settingsModel->get_by_name($settingsModel->unblock_key)->value_text ?? '';
    $unblockLink = (!empty($unblockKey)) ? MainHelper::getUnblockLink($unblockKey): '---';
    $viewManager = new ViewManager();

    echo $viewManager->renderPartial('_popups', 'special_popups', ['unblockLink' => $unblockLink], true);
});

function bw_set_active_submenu($parent_file) {
    global $submenu_file;

    if (isset($_GET['page']) && strpos($_GET['page'], BW_PLUGIN_SLUG) === 0) {
        if (isset($_GET['c']) && $_GET['c'] === 'general_scan') {
            if (isset($_GET['a']) && $_GET['a'] === 'index') {
                $submenu_file = BW_PLUGIN_SLUG . '&c=general_scan&a=index';
            }
        } elseif (isset($_GET['c']) && $_GET['c'] === 'firewall') {
            if (isset($_GET['a']) && $_GET['a'] === 'index') {
                $submenu_file = BW_PLUGIN_SLUG . '&c=firewall&a=index';
            }
            if (isset($_GET['a']) && $_GET['a'] === 'settings') {
                $submenu_file = BW_PLUGIN_SLUG . '&c=firewall&a=settings';
            }
        }
    }

    return $parent_file;
}

function bw_add_dashboard_widget() {
    $initHelper = new InitHelper();
    wp_add_dashboard_widget(
        BW_PLUGIN_SLUG . '_widget',
        __('Betawall Protection Summary', BW_PLUGIN_SLUG),
        [$initHelper, 'init_widget']
    );
}

function bw_plugin_meta($links, $file) {
    if ($file === plugin_basename(__FILE__)) {
        $links[] = '<a href="https://betawall.pro/faq/" target="_blank">' . __('Help', BW_PLUGIN_SLUG) . '</a>';
    }
    return $links;
}

/* start block: Multi-user 2FA */
add_action('wp_ajax_nopriv_mu2fa_check', [$firewallHelper, 'mu2fa_check_callback']);
add_filter('authenticate', [$firewallHelper, 'mu2fa_authenticate_user'], 30, 3);
add_filter('login_message', [$firewallHelper, 'mu2fa_login_message']);
/* end block: Multi-user 2FA */

/* start block: Google reCAPTCHA */
add_action('login_form', [$firewallHelper, 'add_recaptcha_to_login_form']);
add_filter('authenticate', [$firewallHelper, 'verify_recaptcha_on_login'], 99, 3);
//add_action('admin_enqueue_scripts', [$firewallHelper, 'enqueue_recaptcha_script']); // loading JS in admin
/* end block: Google reCAPTCHA */


add_action('login_enqueue_scripts', 'bw_handle_login_enqueue');
function bw_handle_login_enqueue() {
    $firewallHelper = FirewallHelper::getInstance();

    $firewallHelper->check_allowed_ips();
    $firewallHelper->mu2fa_enqueue_login_script();
    $firewallHelper->enqueue_recaptcha_script();
}

